#pragma once

#include <GL/glew.h>

namespace glutil {
	/// wrapper struct for RAII
	struct PushMatrix {
		PushMatrix() { glPushMatrix(); }
		~PushMatrix() { glPopMatrix(); }
	};

	/// wrapper struct for RAII
	struct PushMatrixMode {
		PushMatrixMode(GLenum mode) { glGetIntegerv(GL_MATRIX_MODE, &m_old); glMatrixMode(mode); glPushMatrix(); }
		~PushMatrixMode() { glPopMatrix(); glMatrixMode(m_old); }
	  private:
		GLint m_old;
	};

	/// wrapper struct for RAII
	struct Begin {
		/// call glBegin with given mode
		Begin(GLint mode) { glBegin(mode); }
		~Begin() { glEnd(); }
	};

	/// wrapper struct for RAII
	struct DisplayList {
		/// call glNewList with given list id and mode
		DisplayList(GLuint id, GLenum mode) { glNewList(id, mode); }
		~DisplayList() { glEndList(); }
	};

	/// wrapper struct for RAII
	struct UseLighting {
		/// enable lighting and depth test for 3d objects
		UseLighting(bool doit = true) {
			if (doit) {
				glClear(GL_DEPTH_BUFFER_BIT);
				glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
				
				GLfloat light_position[] = { -50.0, 15.0, -5.0, 1.0 };
				glLightfv(GL_LIGHT0, GL_POSITION, light_position);
				
				glEnable(GL_DEPTH_TEST);
				glEnable(GL_LIGHTING);
				glEnable(GL_COLOR_MATERIAL);
				glEnable(GL_LIGHT0);
			}
		}
		~UseLighting() {
			glDisable(GL_LIGHT0);
			glDisable(GL_COLOR_MATERIAL);
			glDisable(GL_LIGHTING);
			glDisable(GL_DEPTH_TEST);
		}
	};

	/// auto-reversing translation
	struct Translation {
		Translation(float _x, float _y, float _z = 0.0f): m_x(_x), m_y(_y), m_z(_z) 
		  { glTranslatef(m_x, m_y, m_z); }
		~Translation() { glTranslatef(-m_x, -m_y, -m_z); }
	  private:
		float m_x, m_y, m_z;
	};

	/// auto-reversing rotation
	struct Rotation {
		Rotation(float _a, float _x, float _y, float _z): m_a(_a), m_x(_x), m_y(_y), m_z(_z) 
		  { glRotatef(m_a, m_x, m_y, m_z); }
		~Rotation() { glRotatef(-m_a, m_x, m_y, m_z); }
	  private:
		float m_a, m_x, m_y, m_z;
	};

	/// auto-reversing scaling
	struct Scale {
		Scale(float _x, float _y, float _z): m_x(_x), m_y(_y), m_z(_z) 
		  { glScalef(m_x, m_y, m_z); }
		~Scale() { glScalef(1.0/m_x, 1.0/m_y, 1.0/m_z); }
	  private:
		float m_x, m_y, m_z;
	};

	/// struct to store color information
	struct Color {
		float r, ///< red component
		      g, ///< green
		      b, ///< blue
		      a; ///< alpha value
		/// create nec Color object with given channels
		Color(float r_ = 0.0, float g_ = 0.0, float b_ = 0.0, float a_ = 1.0): r(r_), g(g_), b(b_), a(a_) {}
		/// overload float cast
		operator float*() { return reinterpret_cast<float*>(this); }
		/// overload float const cast
		operator float const*() const { return reinterpret_cast<float const*>(this); }
	};
}

