/*
 * Copyright (c) 1995-2006 Silicon Graphics, Inc.  All Rights Reserved.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 */

#include "pmapi.h"
#include "impl.h"
#include "pmda.h"

static int
request_instance (__pmContext *ctxp, pmInDom indom, int inst, const char *name)
{
    int n;

#ifdef ASYNC_API
    if (ctxp->c_pmcd->pc_curpdu != 0) {
	return (PM_ERR_CTXBUSY);
    }
#endif /*ASYNC_API*/
    
    n = __pmSendInstanceReq(ctxp->c_pmcd->pc_fd, __pmPtrToHandle(ctxp),
				&ctxp->c_origin, indom, inst, name);
    if (n < 0) {
	n = __pmMapErrno(n);
    }

    return (n);
}

#ifdef ASYNC_API
static int
ctxid_request_instance (int ctx,  pmInDom indom, int inst, const char *name)
{
    int n;
    __pmContext *ctxp;

    if ((n = __pmGetHostContextByID(ctx, &ctxp)) >= 0) {
	if ((n = request_instance (ctxp, indom, inst, name)) >= 0) {
	    ctxp->c_pmcd->pc_curpdu = PDU_INSTANCE_REQ;
	    ctxp->c_pmcd->pc_tout_sec = TIMEOUT_DEFAULT;
	}
    }

    return (n);
}

int
pmRequestInDomInst (int ctx, pmInDom indom, const char *name)
{
    return (ctxid_request_instance(ctx, indom, PM_IN_NULL, name));
}
#endif /*ASYNC_API*/

static int
receive_instance_id (__pmContext *ctxp)
{
    int n;
    __pmPDU	*pb;

    n = __pmGetPDU(ctxp->c_pmcd->pc_fd, ANY_SIZE, 
		   ctxp->c_pmcd->pc_tout_sec, &pb);
    if (n == PDU_INSTANCE) {
	__pmInResult	*result;
	
	if ((n = __pmDecodeInstance(pb, &result)) >= 0) {
	    n = result->instlist[0];
	    __pmFreeInResult(result);
	}
    }
    else if (n == PDU_ERROR)
	__pmDecodeError(pb, &n);
    else if (n != PM_ERR_TIMEOUT)
	n = PM_ERR_IPC;

    return (n);
}

#ifdef ASYNC_API
int 
pmReceiveInDomInst (int ctx)
{
    int n;
    __pmContext	*ctxp;

    if ((n = __pmGetBusyHostContextByID(ctx, &ctxp, PDU_INSTANCE_REQ)) >= 0) {
	n = receive_instance_id(ctxp);

	ctxp->c_pmcd->pc_curpdu = 0;
	ctxp->c_pmcd->pc_tout_sec = 0;
    }
    return (n);
}
#endif /*ASYNC_API*/

int
pmLookupInDom(pmInDom indom, const char *name)
{
    int		n;
    __pmInResult	*result;
    __pmContext	*ctxp;
    __pmDSO	*dp;

    if (indom == PM_INDOM_NULL)
	return PM_ERR_INDOM;
    if ((n = pmWhichContext()) >= 0) {
	int	ctx = n;
	ctxp = __pmHandleToPtr(ctx);

	if (ctxp->c_type == PM_CONTEXT_HOST) {
	    if ((n = request_instance (ctxp, indom, PM_IN_NULL, name)) >= 0) {
		n = receive_instance_id (ctxp);
	    }
	}
	else if (ctxp->c_type == PM_CONTEXT_LOCAL) {
	    if ((dp = __pmLookupDSO(((__pmInDom_int *)&indom)->domain)) == NULL)
		n = PM_ERR_NOAGENT;
	    else {
		/* We can safely cast away const here */
		if (dp->dispatch.comm.pmda_interface >= PMDA_INTERFACE_5)
		    dp->dispatch.version.four.ext->e_context = ctx;
		if (dp->dispatch.comm.pmda_interface >= PMDA_INTERFACE_4)
		    n = dp->dispatch.version.four.instance(indom, PM_IN_NULL, 
							  (char *)name, 
							  &result, 
							  dp->dispatch.version.four.ext);
		else if (dp->dispatch.comm.pmda_interface == PMDA_INTERFACE_3 ||
		         dp->dispatch.comm.pmda_interface == PMDA_INTERFACE_2)
		    n = dp->dispatch.version.two.instance(indom, PM_IN_NULL, 
							  (char *)name, 
							  &result, 
							  dp->dispatch.version.two.ext);
		else
		    n = dp->dispatch.version.one.instance(indom, PM_IN_NULL, 
							  (char *)name, 
							  &result);
		if (n < 0 && dp->dispatch.comm.pmapi_version == PMAPI_VERSION_1)
		    n = XLATE_ERR_1TO2(n);
	    }
	    if (n >= 0) {
		n = result->instlist[0];
		__pmFreeInResult(result);
	    }
	}
	else {
	    /* assume PM_CONTEXT_ARCHIVE */
	    n = __pmLogLookupInDom(ctxp->c_archctl->ac_log, indom, &ctxp->c_origin, name);
	}
    }

    return n;
}

#ifdef ASYNC_API
int
pmRequestInDomName(int ctx, pmInDom indom, int inst)
{
    return (ctxid_request_instance (ctx, indom, inst, NULL));
}
#endif /*ASYNC_API*/

static int
receive_instance_name(__pmContext *ctxp, char **name)
{
    int n;
    __pmPDU *pb;

    n = __pmGetPDU(ctxp->c_pmcd->pc_fd, ANY_SIZE,
		   ctxp->c_pmcd->pc_tout_sec, &pb);
    if (n == PDU_INSTANCE) {
	__pmInResult *result;

	if ((n = __pmDecodeInstance(pb, &result)) >= 0) {
	    if ((*name = strdup(result->namelist[0])) == NULL)
		n = -oserror();
	    __pmFreeInResult(result);
	}
    }
    else if (n == PDU_ERROR)
	__pmDecodeError(pb, &n);
    else if (n != PM_ERR_TIMEOUT)
	n = PM_ERR_IPC;

    return n;
}

#ifdef ASYNC_API
int
pmReceiveInDomName(int ctx, char **name)
{
    int n;
    __pmContext	*ctxp;

    if ((n = __pmGetBusyHostContextByID(ctx, &ctxp, PDU_INSTANCE_REQ)) >= 0) {
	n = receive_instance_name (ctxp, name);

	ctxp->c_pmcd->pc_curpdu = 0;
	ctxp->c_pmcd->pc_tout_sec = 0;
    }
    return (n);
}
#endif /*ASYNC_API*/

int
pmNameInDom(pmInDom indom, int inst, char **name)
{
    int		n;
    __pmInResult	*result;
    __pmContext	*ctxp;
    __pmDSO	*dp;

    if (indom == PM_INDOM_NULL)
	return PM_ERR_INDOM;
    if ((n = pmWhichContext()) >= 0) {
	int	ctx = n;
	ctxp = __pmHandleToPtr(ctx);
	if (ctxp->c_type == PM_CONTEXT_HOST) {
	    if ((n = request_instance(ctxp, indom, inst, NULL)) >= 0) {
		n = receive_instance_name (ctxp, name);
	    }
	}
	else if (ctxp->c_type == PM_CONTEXT_LOCAL) {
	    if ((dp = __pmLookupDSO(((__pmInDom_int *)&indom)->domain)) == NULL)
		n = PM_ERR_NOAGENT;
	    else {
		if (dp->dispatch.comm.pmda_interface >= PMDA_INTERFACE_5)
		    dp->dispatch.version.four.ext->e_context = ctx;
		if (dp->dispatch.comm.pmda_interface >= PMDA_INTERFACE_4)
		    n = dp->dispatch.version.four.instance(indom, inst, NULL, &result, dp->dispatch.version.four.ext);
		else if (dp->dispatch.comm.pmda_interface == PMDA_INTERFACE_3 ||
		         dp->dispatch.comm.pmda_interface == PMDA_INTERFACE_2)
		    n = dp->dispatch.version.two.instance(indom, inst, NULL, &result, dp->dispatch.version.two.ext);
		else
		    n = dp->dispatch.version.one.instance(indom, inst, NULL, &result);
		if (n < 0 &&
		    dp->dispatch.comm.pmapi_version == PMAPI_VERSION_1)
			n = XLATE_ERR_1TO2(n);
	    }
	    if (n >= 0) {
		if ((*name = strdup(result->namelist[0])) == NULL)
		    n = -oserror();
		__pmFreeInResult(result);
	    }
	}
	else {
	    /* assume PM_CONTEXT_ARCHIVE */
	    char	*tmp;
	    if ((n = __pmLogNameInDom(ctxp->c_archctl->ac_log, indom, &ctxp->c_origin, inst, &tmp)) >= 0) {
		if ((*name = strdup(tmp)) == NULL)
		    n = -oserror();
	    }
	}
    }

    return n;
}

#ifdef ASYNC_API
int
pmRequestInDom (int ctx, pmInDom indom)
{
    return (ctxid_request_instance (ctx, indom, PM_IN_NULL, NULL));
}
#endif /*ASYNC_API*/

static int
inresult_to_lists (__pmInResult *result, int **instlist, char ***namelist)
{
    int n, i, sts, need;
    char *p;
    int *ilist;
    char **nlist;
    
    if (result->numinst == 0) {
	__pmFreeInResult(result);
	return 0;
    }
    need = 0;
    for (i = 0; i < result->numinst; i++) {
	need += sizeof(**namelist) + strlen(result->namelist[i]) + 1;
    }
    ilist = (int *)malloc(result->numinst * sizeof(result->instlist[0]));
    if (ilist == NULL) {
	sts = -oserror();
	__pmFreeInResult(result);
	return sts;
    }
    if ((nlist = (char **)malloc(need)) == NULL) {
	sts = -oserror();
	free(ilist);
	__pmFreeInResult(result);
	return sts;
    }

    *instlist = ilist;
    *namelist = nlist;
    p = (char *)&nlist[result->numinst];
    for (i = 0; i < result->numinst; i++) {
	ilist[i] = result->instlist[i];
	strcpy(p, result->namelist[i]);
	nlist[i] = p;
	p += strlen(result->namelist[i]) + 1;
    }
    n = result->numinst;
    __pmFreeInResult(result);
    return n;
}

int
receive_indom (__pmContext *ctxp,  int **instlist, char ***namelist)
{
    int n;
    __pmPDU	*pb;

    n = __pmGetPDU(ctxp->c_pmcd->pc_fd, ANY_SIZE,
		   ctxp->c_pmcd->pc_tout_sec, &pb);
    if (n == PDU_INSTANCE) {
	__pmInResult	*result;
	    
	if ((n = __pmDecodeInstance(pb, &result)) < 0)
	    return n;

	n = inresult_to_lists (result, instlist, namelist);
    }
    else if (n == PDU_ERROR)
	__pmDecodeError(pb, &n);
    else if (n != PM_ERR_TIMEOUT)
	n = PM_ERR_IPC;

    return (n);
}

#ifdef ASYNC_API
int
pmReceiveInDom (int ctx,  int **instlist, char ***namelist)
{
    int n;
    __pmContext	*ctxp;

    if ((n = __pmGetBusyHostContextByID(ctx, &ctxp,  PDU_INSTANCE_REQ)) >= 0) {
	n = receive_indom (ctxp, instlist, namelist);

	ctxp->c_pmcd->pc_curpdu = 0;
	ctxp->c_pmcd->pc_tout_sec = 0;
    }
    return (n);
}
#endif /*ASYNC_API*/

int
pmGetInDom(pmInDom indom, int **instlist, char ***namelist)
{
    int			n;
    int			i;
    __pmInResult	*result;
    __pmContext		*ctxp;
    char		*p;
    int			need;
    int			*ilist;
    char		**nlist;
    __pmDSO		*dp;

    /* avoid ambiguity when no instances or errors */
    *instlist = NULL;
    *namelist = NULL;
    if (indom == PM_INDOM_NULL)
	return PM_ERR_INDOM;

    if ((n = pmWhichContext()) >= 0) {
	int	ctx = n;
	ctxp = __pmHandleToPtr(ctx);
	if (ctxp->c_type == PM_CONTEXT_HOST) {
	    if ((n = request_instance (ctxp, indom, PM_IN_NULL, NULL)) >= 0) {
		n = receive_indom (ctxp, instlist, namelist);
	    }
	}
	else if (ctxp->c_type == PM_CONTEXT_LOCAL) {
	    if ((dp = __pmLookupDSO(((__pmInDom_int *)&indom)->domain)) == NULL)
		n = PM_ERR_NOAGENT;
	    else {
		if (dp->dispatch.comm.pmda_interface >= PMDA_INTERFACE_5)
		    dp->dispatch.version.four.ext->e_context = ctx;
		if (dp->dispatch.comm.pmda_interface >= PMDA_INTERFACE_4)
		    n = dp->dispatch.version.four.instance(indom, PM_IN_NULL, NULL,
					       &result,
					       dp->dispatch.version.four.ext);
		else if (dp->dispatch.comm.pmda_interface == PMDA_INTERFACE_3 ||
		         dp->dispatch.comm.pmda_interface == PMDA_INTERFACE_2)
		    n = dp->dispatch.version.two.instance(indom, PM_IN_NULL, NULL,
					       &result,
					       dp->dispatch.version.two.ext);
		else
		    n = dp->dispatch.version.one.instance(indom, PM_IN_NULL, NULL,
					      &result);
		if (n < 0 &&
		    dp->dispatch.comm.pmapi_version == PMAPI_VERSION_1)
			n = XLATE_ERR_1TO2(n);
	    }
	    if (n >= 0) {
		n = inresult_to_lists (result, instlist, namelist);
	    }
	}
	else {
	    /* assume PM_CONTEXT_ARCHIVE */
	    int		*insttmp;
	    char	**nametmp;
	    if ((n = __pmLogGetInDom(ctxp->c_archctl->ac_log, indom, &ctxp->c_origin, &insttmp, &nametmp)) >= 0) {
		need = 0;
		for (i = 0; i < n; i++) {
		    need += sizeof(char *) + strlen(nametmp[i]) + 1;
		}
		if ((ilist = (int *)malloc(n * sizeof(insttmp[0]))) == NULL) {
		    return -oserror();
		}
		if ((nlist = (char **)malloc(need)) == NULL) {
		    free(ilist);
		    return -oserror();
		}
		*instlist = ilist;
		*namelist = nlist;
		p = (char *)&nlist[n];
		for (i = 0; i < n; i++) {
		    ilist[i] = insttmp[i];
		    strcpy(p, nametmp[i]);
		    nlist[i] = p;
		    p += strlen(nametmp[i]) + 1;
		}
	    }
	}
    }

    return n;
}

#ifdef PCP_DEBUG
void
__pmDumpInResult(FILE *f, const __pmInResult *irp)
{
    int		i;
    fprintf(f,"pmInResult dump from " PRINTF_P_PFX "%p for InDom %s (0x%x), numinst=%d\n",
		irp, pmInDomStr(irp->indom), irp->indom, irp->numinst);
    for (i = 0; i < irp->numinst; i++) {
	fprintf(f, "  [%d]", i);
	if (irp->instlist != NULL)
	    fprintf(f, " inst=%d", irp->instlist[i]);
	if (irp->namelist != NULL)
	    fprintf(f, " name=\"%s\"", irp->namelist[i]);
	fputc('\n', f);
    }
    return;
}
#endif

void
__pmFreeInResult(__pmInResult *res)
{
    int		i;

    if (res->namelist != NULL) {
	for (i = 0; i < res->numinst; i++) {
	    if (res->namelist[i] != NULL) {
		free(res->namelist[i]);
	    }
	}
	free(res->namelist);
    }
    if (res->instlist != NULL)
	free(res->instlist);
    free(res);
}
