# Copyright (C) 2003-2005 The Perl Foundation.  All rights reserved.
# $Id: Long.pir 10743 2005-12-28 20:48:15Z particle $

=head1 NAME

library/Getopt/Long.pir - parse long and short command line options

=head1 SYNOPSIS

  # get the relevant sub
  load_bytecode "Getopt/Long.pbc"
  .local pmc get_options
  find_global get_options, "Getopt::Long", "get_options" 

  # Assemble option specification
  .local pmc opt_spec
  opt_spec = new ResizableStringArray
  push opt_spec, "bool"
  push opt_spec, "string=s"
  push opt_spec, "integer=i"

  # the program name is the first element in argv
  .local string program_name
  program_name = shift argv

  # Parse the command line params
  .local pmc opt
  ( opt ) = get_options( argv, opt_spec )

  .local int is_defined
  is_defined = defined opt["bool"]

  .local int integer
  integer = opt["integer"]

  .local string s
  s = opt["string"]

=head1 DESCRIPTION

This Parrot library can be used for parsing command line options.
The subroutine get_options() is exported into the namespace 'Getopt::Long'.
Everything after '--' is not regarded as an option.
Options and additional parameters cannot be mixed. Options come first.

=head1 SUBROUTINES

=head2 C<get_options> in C<Getopt::Long>

This should work like the Perl5 module Getopt::Long.
Takes an array of options and an array of specifications.

A Hash PMC is returned.

=head1 TODO

- Make it work for all cases, short options, long options and bundling.
- Recognise type of return value: string, integer, binary, array, hash.
- Get started on error reporting.
- Provide more options

=head1 AUTHOR

Bernhard Schmalhofer - C<Bernhard.Schmalhofer@gmx.de>

=head1 SEE ALSO

The Perl5 module L<Getopt::Long>.
F<examples/library/getopt_demo.pir>
F<t/library/getopt_long.t>

=head1 COPYRIGHT

Copyright (C) 2003-2005 The Perl Foundation.  All rights reserved.
This program is free software. It is subject to the same
license as The Parrot Interpreter.

=cut

.include "library/dumper.pir"
 
.namespace [ "Getopt::Long" ]

.sub get_options 
  # TODO: Check whether these are really arrays  
  .param pmc argv                    # An Array containing command line args
  .param pmc spec                    # An Array containing the spec

  # Loop over the array spec and build up two simple hashes
  .local pmc    type                 # the type of the option: binary, string, integer
  type = new Hash
  .local int    curr_spec            # a counter for looping over the array 'spec'
  curr_spec = 0
  .local int    max_spec             # for end condition of loop over 'spec'
  max_spec = spec
  .local int    spec_index           # searching for patterns in 'spec'
  .local string opt_name, opt_type   # name and type of specified option

  # TODO: put this block into a sub
  goto CHECK_PARSE_SPEC
  NEXT_PARSE_SPEC:                   # Look at next element in 'spec'
    opt_name = spec[curr_spec]
    # opt_name is stored into the hash - this creates reference
    # the substr below modifies this hash key in place, so we
    # have to clone it
    opt_name = clone opt_name
    
    spec_index = index opt_name, "="
    if spec_index != -1 goto NOT_A_BINARY_OPTION
    opt_type = 'b'
    goto OPTION_TYPE_IS_NOW_KNOWN
  NOT_A_BINARY_OPTION:
    inc spec_index                   # we know where '=', thus the type is one further
    opt_type = substr opt_name, spec_index, 1, ''
    dec spec_index                   # Go back to the '='
    # TODO: what if we have something like    name=xy ?
    substr opt_name, spec_index, 1, ''   # remove the '='
  OPTION_TYPE_IS_NOW_KNOWN:
    type[opt_name] = opt_type
    inc curr_spec
  CHECK_PARSE_SPEC:                  # check whether loop over 'spec' is complete
  if curr_spec < max_spec goto NEXT_PARSE_SPEC

  # uncomment this if you want debug output
  goto SKIP_DEBUG_OUTPUT
  _dumper( 'type', type )
  SKIP_DEBUG_OUTPUT:

  # Now that we know about the allowed options,
  # we actually parse the argument vector
  # shift from argv until a non-option is encountered
  .local pmc opt              # the return PMC
  opt = new Hash
  .local string arg                  # element of argument array
  .local string value                # element of argument array
  .local int    num_remaining_args   # for checking whether loop is complete
  .local int    arg_index            # holds result if 'index' op
  .local int    is_known_option      # flag whether the option is known
  .local int    prefix_end, prefix_len
  goto CHECK_PARSE_ARGV
  NEXT_PARSE_ARGV:
    # first we take a peek at the first remaining element
    arg = argv[0]
    # arg is stored into the hash - this creates reference
    # the substr below modifies this hash key in place, so we
    # have to clone it
    arg = clone arg

    # Is arg the option terminator '--'?
    if arg  == "--" goto HANDLE_OPTION_TERMINATOR

    # Is arg a long option string like '--help'?
    $S0 = substr arg, 0, 2
    if $S0 == "--" goto HANDLE_LONG_OPTION

    # Is arg a short option string like '-v'?
    $S0 = substr arg, 0, 1 
    if $S0 == "-" goto HANDLE_SHORT_OPTION

    # We are done with the option
    # and we don't want to loose the remaining arguments
    goto FINISH_PARSE_ARGV

    HANDLE_OPTION_TERMINATOR:
    # The '--' is not part of the remaining options
    arg = shift argv
    goto FINISH_PARSE_ARGV

    HANDLE_SHORT_OPTION:
    arg = shift argv
    arg = clone arg
    # get rid of the leading '-'
    substr arg, 0, 1, ''
    goto GET_VALUE
    
    HANDLE_LONG_OPTION:
    arg = shift argv
    arg = clone arg
    # get rid of the leading '--'
    substr arg, 0, 2, ''

    GET_VALUE:
    # recover the value if any
    arg_index = index arg, "="
    if arg_index != -1 goto VALUE_PASSED
    opt[arg] = 1
    goto VALUE_OF_OPTION_IS_NOW_KNOWN
    VALUE_PASSED:
    inc arg_index    # Go one past the '='
    .local int len_value
    len_value = length arg
    len_value = len_value - arg_index
    value = substr arg, arg_index, len_value
    # drop the '=file.m4' from '--freeze-state=file.m4'
    dec arg_index
    inc len_value
    arg = substr arg_index, len_value, ''
    opt[arg] = value
    VALUE_OF_OPTION_IS_NOW_KNOWN:
    # Is this a known option?
    # TODO: make this work for nonbinary options
    is_known_option = defined type[arg]
    unless is_known_option goto UNKNOWN_OPTION
    # Tell the caller that the option 'arg' has been passed
    goto CHECK_PARSE_ARGV
    UNKNOWN_OPTION:
    # TODO: handle unknown options
    printerr 'unknown option: !'
    printerr  arg
    printerr "!\n"

  CHECK_PARSE_ARGV:
    num_remaining_args = argv
    if num_remaining_args > 0 goto NEXT_PARSE_ARGV

  FINISH_PARSE_ARGV:
  # Nothing to do 

  .return ( opt )
.end
