#ifdef WIN32
#include <windows.h>
#include <io.h>
#endif

#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>

#include "papaya/system.h"
#include "mudclient.h"

#include "interface.h"
#include "support.h"

#include "Socket.h"

#include "Connection.h"
#include "Prefs.h"
#include "Message.h"

#include "callbacks.h"
#include "PluginHandler.h"
#include "EntityHandler.h"
#include "PreferenceHandler.h"
#include "ProxyPreference.h"
#include "PromptPreference.h"
#include "HistoryPreference.h"
#include "InputPreference.h"
#include "OutputPreference.h"
#ifdef ZLIB
#include "CompressionPreference.h"
#endif // ZLIB
#include "ConnectionPreference.h"
#include "FontPreference.h"
#include "ColourPreference.h"

#include "MUD.h"
#include "MUDSelector.h"
#include "MUDConnectorList.h"

#include "GTKTwoWindow.h"

#include "libusers.h"

#define PACKAGE "papaya"

MUDSelector * mud_selector;
MUDConnectorList * mcl;
BaseWindow * mainWindow;
EntityHandler * entities;
PluginHandler * phandler;
PreferenceHandler * preferenceHandler;
Prefs * globalPreferences;

void debug(char * str, ...) {

  // We don't want to display debug messages in a release build.
  return;

  char buf[16384];
  va_list ap;

  va_start(ap, str);
#ifdef WIN32
#define vsnprintf _vsnprintf
#endif
  vsnprintf(buf, 16384, str, ap);
  va_end(ap);

  fprintf(stderr, "%s", buf);
}

#include <glade/glade.h>

static GladeXML * splash_xml = NULL;

void splash_create() {
  char buf[1024];

  snprintf(buf, 1024, "%s/share/papaya/splash.glade", getPrefix());

  splash_xml = glade_xml_new(buf, NULL, NULL);
  if (!splash_xml) {
    printf ("Could not load splash screen.  Aborting.\n");
    abort();
  }
}

void splash_destroy() {
  GtkWidget * splash_window = glade_xml_get_widget(splash_xml, "splash_window");
  gtk_widget_destroy(splash_window);

  g_object_unref(splash_xml);
  splash_xml = NULL;
}

void splash_update(gchar * text, ...) {
  /*
  if (!globalPreferences->getUseSplashScreen())
    return;
  */

  if (!splash_xml)
    return;

  GtkWidget * splash_window = glade_xml_get_widget(splash_xml, "splash_window");

  if (!splash_window)
    return;

  char buf[16384];
  va_list ap;

  va_start(ap, text);
  vsnprintf(buf, 16384, text, ap);
  va_end(ap);

  GtkWidget * splash_progressbar = glade_xml_get_widget(splash_xml, "splash_progress_bar");

  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(splash_progressbar), buf);  

  while (gtk_events_pending())
    gtk_main_iteration();
}

int papaya_startup(int argc, char * argv[]) {

  bool copyover = false;

  char buf[1024];

  snprintf(buf, 1024, "%s/share/papaya", getPrefix());

  add_pixmap_directory("./");
  add_pixmap_directory("bitmaps");
  add_pixmap_directory(buf);

#ifndef WIN32
  // Ensure that we have a $HOME/.papaya/ directory if possible.
  char * home = getenv("HOME");
  if (home) {
    char filename[1024];
    snprintf(filename, 1024, "%s/.papaya", home);
    mkdir(filename, 0700);
  }
#endif

#ifdef ENABLE_NLS
  setlocale(LC_ALL, "");
  snprintf(buf, 1024, "%s/share/locale", getPrefix());
  bindtextdomain(PACKAGE, buf);

  // Attempt to force output in UTF-8 - THIS IS NECESSARY!
  bind_textdomain_codeset(PACKAGE, "UTF-8");
  textdomain(PACKAGE);
#endif // ENABLE_NLS

  gtk_init(&argc, &argv);

#if defined(sun)
  sigignore(SIGPIPE);
#endif

#ifndef WIN32
  signal(SIGCHLD, SIG_IGN);
  signal(SIGPIPE, SIG_IGN);
#endif

  struct stat stat_buf;
#if defined(WIN32) && !defined(__MINGW32__)
  snprintf(buf, 1024, "gtkrc");
  if (stat(buf, &stat_buf) != 0) {
    snprintf(buf, 1024, "%s/share/papaya/gtkrc", getPrefix());
  }
#else
#  if !defined(__MINGW32__)
  home = getenv("HOME");
  snprintf(buf, 1024, "%s/.papaya/gtkrc", home);
  if (stat(buf, &stat_buf) != 0) {
#  endif
    snprintf(buf, 1024, "%s/share/papaya/gtkrc", getPrefix());
#  if !defined(__MINGW32__)
  }
#  endif
#endif
  gtk_rc_parse(buf);
    
  // Load up the preference handler.
  preferenceHandler = new PreferenceHandler();
  preferenceHandler->addPreferenceEditor(new ProxyPreference());
  preferenceHandler->addPreferenceEditor(new PromptPreference());
  preferenceHandler->addPreferenceEditor(new HistoryPreference());
  preferenceHandler->addPreferenceEditor(new InputPreference());
  preferenceHandler->addPreferenceEditor(new OutputPreference());
#ifdef ZLIB
  preferenceHandler->addPreferenceEditor(new CompressionPreference());
#endif
  preferenceHandler->addPreferenceEditor(new ConnectionPreference());
  preferenceHandler->addPreferenceEditor(new FontPreference());
  preferenceHandler->addPreferenceEditor(new ColourPreference());

  // Load in the global preferences so we know if the user wants a splash
  // screen or not.
  globalPreferences = new Prefs((MUD *)NULL);

  // Create the splash screen
  splash_create();

#ifdef WIN32
  splash_update(_("Initialising network"));
  WSADATA wsadata;
  if (WSAStartup(MAKEWORD(2,2), &wsadata) != 0) {
    new Message("Error", "Failed to initialise Windows Sockets.", true);
    exit(1);
  }
#endif
  
  splash_update(_("Loading MUD Connection Wizard data"));
  mud_selector = new MUDSelector();

  splash_update(_("Loading MUD Connector MUD list"));
  mcl = new MUDConnectorList();
  
  splash_update(_("Creating main window"));
  mainWindow = new GTKTwoWindow();

  splash_update(_("Loading Aliases and Triggers"));
  entities = new EntityHandler();
  entities->load();

  splash_update(_("Loading Plugins from %s"), getPrefix());
  phandler = new PluginHandler();
  phandler->loadPlugins();

  splash_destroy();

  mainWindow->display();
  for (int i = 0; i < argc; i++) {
    if (!strcmp(argv[i], "copyover")) {
      copyover_recover();
      copyover = TRUE;
      break;
    }
  }

  on_connection_wizard_button_clicked((GtkButton *)NULL, (gpointer)NULL, 0);

  gtk_main();
  return 1;
}

int papaya_shutdown() {

#if 0
  if (globalPreferences->getSendStats()) {
    int count_socket = create_socket();
    disconnect_from_server(count_socket);
    close(count_socket);
  }
#endif

#if !defined(sun) || !defined(WIN32)
  signal(SIGSEGV, NULL);
#endif

  delete mainWindow;
  delete phandler;
  delete globalPreferences;
  delete mud_selector;
  delete mcl;

  entities->save();
  delete entities;

#ifdef WIN32
  WSACleanup();
#endif

  return 0;
}
