/**
 * Plugin class, extended by all plugins.
 *
 * If this structure is not modified AT ALL, but the implementation in Plugin.cpp
 * is, then PLUGIN_API_BUGFIX should be incremented by 1.  If the structure is
 * altered in such a way that plugins already compiled against the old version of
 * the structure will still work, increment PLUGIN_API_MINOR by 1.  If the structure
 * is changed in any other way, increment PLUGIN_API_MAJOR by 1.
 *
 * The API version number has no bearing on Papaya version, or Plugin version.
 */

#ifndef PLUGIN_H
#define PLUGIN_H

#define PLUGIN_API_MAJOR 1
#define PLUGIN_API_MINOR 0
#define PLUGIN_API_BUGFIX 0

#include <gmodule.h>
#include <stdio.h>

#include "papaya/system.h"

class Connection;
class Event;
class MUD;

class Plugin {
 public:

  Plugin();
  virtual ~Plugin();

  virtual void input(Connection * c, char * in);
  virtual void output(Connection * c, char * in);
  virtual void prompt(Connection * c, char * in);
  virtual void clientMessage(Connection * c, char * in);

  virtual void teloptSubneg(Connection * c, int option, char *, int);
  virtual bool teloptAllowLocal(Connection * c, int option);
  virtual bool teloptAllowRemote(Connection * c, int option);
  virtual void teloptHandleRemote(Connection * c, int option, bool);
  
  virtual void onEvent(Event * e, Connection * c);

  virtual void pageSwitched(); // When the user changes pages in the GTKTwoWindow.


  virtual char * getDescription();

  char * getName() {return name;};
  float getVersion() {return version;};

  virtual void onPrefsApply(MUD *);
  virtual void onPrefsOk(MUD *);
  virtual void savePrefs(FILE *, MUD *);
  virtual void onPrefsCancel(MUD *);

 protected:

  float version;
  char * name;
};



#if defined(PLUGIN)  && !defined(PLUGIN_CPP)
// DO NOT OVERRIDE
extern "C" int G_MODULE_EXPORT plugin_query_api_major() {
  return PLUGIN_API_MAJOR;
}
// DO NOT OVERRIDE
extern "C" int G_MODULE_EXPORT plugin_query_api_minor() {
  return PLUGIN_API_MINOR;
}
// DO NOT OVERRIDE
extern "C" int G_MODULE_EXPORT plugin_query_api_bugfix() {
  return PLUGIN_API_BUGFIX;
}
// DO NOT OVERRIDE
extern "C" int G_MODULE_EXPORT plugin_is_valid() {
  return 1;
}
#endif // -DPLUGIN

#endif
