//=======================================================================
// mainwindow.cc
//-----------------------------------------------------------------------
// This file is part of the package paco
// Copyright (C) 2004-2009 David Rosal
// For more information visit http://paco.sourceforge.net
//=======================================================================

#include "config.h"
#include "gconfig.h"
#include "mainwindow.h"
#include "preferences.h"
#include "about.h"
#include "pkgset.h"
#include "pkg.h"	// TAB_*
#include "find.h"
#include "util.h"
#include "paco/common.h"	// toString()
#include <gtkmm/stock.h>
#include <gtkmm/toggleaction.h>
#include <gtkmm/uimanager.h>
#include <gtkmm/actiongroup.h>
#include <gtkmm/box.h>
#include <gtkmm/separator.h>
#include <gtkmm/scrolledwindow.h>
#include <sstream>

using Glib::ustring;
using std::string;
using namespace sigc;
using namespace Gpaco;


MainWindow::MainWindow()
:
	Gtk::Window(),
	mLabel("", 0.02, 0.5),
	mTreeView(),
	mProgressBar(),
	mpUIManager(Gtk::UIManager::create()),
	mpActionGroup(Gtk::ActionGroup::create()),
	mpActionUpdate(Gtk::Action::create("UpdateDatabase", Gtk::Stock::REFRESH, "_Update database")),
	mpActionViewFiles(Gtk::Action::create("ViewFiles", Gtk::Stock::DIRECTORY, "View _files")),
	mpActionPkgInfo(Gtk::Action::create("PkgInfo", Gtk::Stock::INFO)),
	mpActionRemovePkg(Gtk::Action::create("RemovePkg", Gtk::Stock::DELETE, "_Remove...")),
	mpActionCreatePackage(Gtk::Action::create("CreatePackage", Gtk::Stock::EXECUTE, "Create _package...")),
	mpActionUnlogPkg(Gtk::Action::create("UnlogPkg", Gtk::Stock::REMOVE, "Remove from _database..."))
{
	set_default_size(GConfig::width(), GConfig::height());
	try { set_icon_from_file(DATADIR "/pixmaps/gpaco.png"); }
	catch (...) { }

	buildUIManager();

	for (guint i = 0; i < GConfig::cols().size(); ++i)
		mTreeView.get_column(GConfig::cols()[i])->set_visible();
	
	Gtk::ScrolledWindow* pScrolledWindow(Gtk::manage(new Gtk::ScrolledWindow()));
	pScrolledWindow->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	pScrolledWindow->add(mTreeView);

	mProgressBar.set_size_request(-1, 0);

	Gtk::HBox* pHBox(Gtk::manage(new Gtk::HBox()));
	pHBox->pack_start(mLabel, Gtk::PACK_SHRINK, 8);
	pHBox->pack_end(mProgressBar, Gtk::PACK_SHRINK);

	mpUIManager->get_widget("/MenuBar/MenuPackage")->signal_event().connect(mem_fun(*this, &MainWindow::onPkgMenu));

	Gtk::VBox* pVBox(Gtk::manage(new Gtk::VBox()));
	pVBox->pack_start(*(mpUIManager->get_widget("/MenuBar")), Gtk::PACK_SHRINK);
	pVBox->pack_start(*pScrolledWindow);
	pVBox->pack_start(*(Gtk::manage(new Gtk::HSeparator())), Gtk::PACK_SHRINK, 1);
	pVBox->pack_start(*pHBox, Gtk::PACK_SHRINK);

	add(*pVBox);
	show_all_children();
	mProgressBar.hide();
	move(GConfig::xPos(), GConfig::yPos());

	writeLabel();
	Glib::signal_timeout().connect(mem_fun(*this, &MainWindow::writeLabel), 500);
}


MainWindow::~MainWindow()
{ }


bool MainWindow::onPkgMenu(GdkEvent*)
{
	int rows = mTreeView.countSelected();
	
	mpActionViewFiles		->set_sensitive(rows == 1);
	mpActionPkgInfo			->set_sensitive(rows == 1);
	mpActionRemovePkg		->set_sensitive(rows == 1 && GConfig::logdirWritable());
	mpActionCreatePackage	->set_sensitive(rows == 1);
	mpActionUnlogPkg		->set_sensitive(rows && GConfig::logdirWritable());

	return false;
}


// [virtual]
bool MainWindow::on_delete_event(GdkEventAny*)
{
	quit();
	return true;
}


void MainWindow::quit()
{
	GConfig::saveSettings(*this);
	Glib::usleep(10000);
	hide();
}


void MainWindow::buildUIManager()
{
	g_assert(mpActionGroup);
	g_assert(mpActionUpdate);

	mpActionGroup->add(Gtk::Action::create("MenuFile", "_File"));
	mpActionGroup->add(mpActionUpdate, mem_fun(mTreeView, &MainTreeView::onUpdateDataBase));
	mpActionUpdate->set_sensitive(GConfig::logdirWritable());
	mpActionGroup->add(Gtk::Action::create("Quit", Gtk::Stock::QUIT),
		mem_fun(*this, &MainWindow::quit));
		
	mpActionGroup->add(Gtk::Action::create("MenuEdit", "_Edit"));
	mpActionGroup->add(Gtk::Action::create("FindFile", Gtk::Stock::FIND, "_Find file"),
		ptr_fun(&Find::instance));
	mpActionGroup->add(Gtk::Action::create("Preferences", Gtk::Stock::PREFERENCES),
		ptr_fun(&Preferences::instance));
	mpActionGroup->add(Gtk::Action::create("SelectAll", "Select _all"),
		mem_fun(mTreeView, &TreeView::selectAll));
	mpActionGroup->add(Gtk::Action::create("UnselectAll", "_Unselect all"),
		mem_fun(mTreeView, &TreeView::unselectAll));
	
	mpActionGroup->add(Gtk::Action::create("MenuPackage", "_Package"));
	mpActionGroup->add(mpActionViewFiles, bind<int>(mem_fun(mTreeView, &MainTreeView::onPkgWindow), TAB_FILES));
	mpActionGroup->add(mpActionPkgInfo, bind<int>(mem_fun(mTreeView, &MainTreeView::onPkgWindow), TAB_INFO));
	mpActionGroup->add(mpActionRemovePkg, bind<int>(mem_fun(mTreeView, &MainTreeView::onPkgWindow), TAB_REMOVE));
	mpActionGroup->add(mpActionCreatePackage, bind<int>(mem_fun(mTreeView, &MainTreeView::onPkgWindow), TAB_PACKAGE));
	mpActionGroup->add(mpActionUnlogPkg, mem_fun(mTreeView, &MainTreeView::onUnlog));

	mpActionGroup->add(Gtk::Action::create("MenuHelp", "_Help"));
	mpActionGroup->add(Gtk::Action::create("About", Gtk::Stock::ABOUT), ptr_fun(&About::instance));

	mpUIManager->insert_action_group(mpActionGroup);
	
	add_accel_group(mpUIManager->get_accel_group());

	mpUIManager->add_ui_from_string(
		"<ui>"
		"	<menubar name='MenuBar'>"
		"		<menu action='MenuFile'>"
		"			<menuitem action='UpdateDatabase'/>"
		"			<separator/>"
		"			<menuitem action='Quit'/>"
		"		</menu>"
		"		<menu action='MenuEdit'>"
		"			<menuitem action='FindFile'/>"
		"			<separator/>"
		"			<menuitem action='Preferences'/>"
		"			<separator/>"
		"			<menuitem action='SelectAll'/>"
		"			<menuitem action='UnselectAll'/>"
		"		</menu>"
		"		<menu action='MenuPackage'>"
		"			<menuitem action='ViewFiles'/>"
		"			<menuitem action='PkgInfo'/>"
		"			<menuitem action='RemovePkg'/>"
		"			<menuitem action='CreatePackage'/>"
		"			<separator/>"
		"			<menuitem action='UnlogPkg'/>"
		"		</menu>"
		"		<menu action='MenuHelp'>"
		"			<menuitem action='About'/>"
		"		</menu>"
		"	</menubar>"
		"</ui>");
}


bool MainWindow::isColumnVisible(int colId)
{
	return mTreeView.get_column(colId)->get_visible();
}


void MainWindow::setColumnVisible(int colId, bool visible /* = true */)
{
	mTreeView.get_column(colId)->set_visible(visible);
}


bool MainWindow::writeLabel()
{
	if (mProgressBar.is_visible()) {
		// Updating the database
		return true;
	}

	std::ostringstream txt;
	txt << mTreeView.pkgSet().size() << " Packages  |  "
		<< Paco::toString(mTreeView.pkgSet().sizeInst());
	
	mLabel.set_text(txt.str());
	
	return true;
}

