//=======================================================================
// gconfig.cc
//-----------------------------------------------------------------------
// This file is part of the package paco
// Copyright (C) 2004-2009 David Rosal
// For more information visit http://paco.sourceforge.net
//=======================================================================

#include "config.h"
#include "util.h"
#include "gconfig.h"
#include "mainwindow.h"
#include <fstream>

using std::string;
using std::vector;
using namespace Gpaco;

namespace Gpaco
{
	bool GConfig::sHour	= false;
	int GConfig::sWidth	= GConfig::DEFAULT_WIDTH;
	int GConfig::sHeight	= GConfig::DEFAULT_HEIGHT;
	int GConfig::sXPos	= GConfig::DEFAULT_XPOS;
	int GConfig::sYPos	= GConfig::DEFAULT_YPOS;
	vector<int> GConfig::sCols;
}


GConfig::GConfig()
:
	Paco::BaseConfig(),
	mGpacoRC(Glib::get_home_dir() + "/.config/gpaco/gpacorc"),
	mKeyFile(g_key_file_new())
{
	if (!checkLogdir())
		errorDialog(NULL, sLogdir + ": Invalid log directory", true);

	sCols.push_back(MainTreeView::COL_ICON);
	sCols.push_back(MainTreeView::COL_NAME);
	sCols.push_back(MainTreeView::COL_SIZE_INST);
	sCols.push_back(MainTreeView::COL_SIZE_PERCENT);
	sCols.push_back(MainTreeView::COL_DATE);
	sCols.push_back(MainTreeView::COL_SUMMARY);

	string dir(Glib::get_home_dir() + "/.config");
	mkdir(dir.c_str(), 0755);
	dir += "/gpaco";
	mkdir(dir.c_str(), 0755);

	if (!g_key_file_load_from_file(mKeyFile, mGpacoRC.c_str(), G_KEY_FILE_NONE, NULL))
		return;

	// Remove gpacorc if is of an older version (< 2)
	gchar* version = g_key_file_get_string(mKeyFile, "settings", "VERSION", NULL);
	
	if (!version) {
		g_key_file_free(mKeyFile);
		unlink(mGpacoRC.c_str());
		return;
	}
	g_free(version);

	g_return_if_fail(g_key_file_has_key(mKeyFile, "settings", "COLUMNS", NULL));
	
	getValue("SHOW_HOUR", &sHour);
	getValue("WINDOW_WIDTH", &sWidth);
	getValue("WINDOW_HEIGHT", &sHeight);
	getValue("WINDOW_XPOS", &sXPos);
	getValue("WINDOW_YPOS", &sYPos);

	GError* e = NULL;
	gsize n;
	int* columns = g_key_file_get_integer_list(mKeyFile, "settings", "COLUMNS", &n, &e);
	
	if (!e && columns && n) {
		sCols.assign(columns, columns + n);
		g_free(columns);
	}

	g_key_file_free(mKeyFile);
}


GConfig::~GConfig()
{
	mKeyFile = g_key_file_new();
	g_key_file_set_string(mKeyFile, "settings", "VERSION", PACKAGE_VERSION);
	g_key_file_set_boolean(mKeyFile, "settings", "SHOW_HOUR", sHour);
	g_key_file_set_integer(mKeyFile, "settings", "WINDOW_WIDTH", sWidth);
	g_key_file_set_integer(mKeyFile, "settings", "WINDOW_HEIGHT", sHeight);
	g_key_file_set_integer(mKeyFile, "settings", "WINDOW_XPOS", sXPos);
	g_key_file_set_integer(mKeyFile, "settings", "WINDOW_YPOS", sYPos);

	int n = sCols.size();
	int* colList = new int[n];
	for (int i = 0; i < n; ++i)
		colList[i] = sCols[i];
	g_key_file_set_integer_list(mKeyFile, "settings", "COLUMNS", colList, n);
	delete [] colList;

	std::ofstream f(mGpacoRC.c_str());
	if (f)
		f << g_key_file_to_data(mKeyFile, 0, 0);
	else
		g_warning("Cannot open configuration file %s for writing\n", mGpacoRC.c_str());

	g_key_file_free(mKeyFile);
}


void GConfig::saveSettings(MainWindow& window)
{
	window.get_size(sWidth, sHeight);
	window.get_position(sXPos, sYPos);

	sCols.clear();
	for (int i = 0; i < MainTreeView::N_COLS; ++i) {
		if (window.isColumnVisible(i))
			sCols.push_back(i);
	}
}


// [static]
void GConfig::init()
{
	static GConfig c;
}


void GConfig::getValue(char const* tag, bool* var)
{
	GError* error = 0;
	bool val = g_key_file_get_boolean(mKeyFile, "settings", tag, &error);
	if (!error)
		*var = val;
}


void GConfig::getValue(char const* tag, int* var)
{
	GError* error = 0;
	int val = g_key_file_get_integer(mKeyFile, "settings", tag, &error);
	if (!error)
		*var = val;
}

