#include "aptpackagedb.h"

#include "aptpackage.h"

// NPlugin
#include "packagenotfoundexception.h"

#include <iprogressobserver.h>

#include <QtDebug>

namespace NApt {


AptPackageDB::AptPackageDB(NUtil::IProgressObserver *pObserver) :
    _pCacheFile(new pkgCacheFile()),
    _pPkgCache(_pCacheFile->GetPkgCache()),
    _pRecs(new pkgRecords(*_pPkgCache))
{
    initializePackages(pObserver);
}

AptPackageDB::~AptPackageDB()
{
    deletePackages();
}


void AptPackageDB::initializePackages(NUtil::IProgressObserver *pObserver) {
    if (pObserver) pObserver->setProgress(0);
    _pCacheFile = new pkgCacheFile;
    _pPkgCache = _pCacheFile->GetPkgCache();
    _pRecs = new pkgRecords(*_pPkgCache);
    double packageCountEstimate = 250000.0; // informed guess :o
    long progress = 0;
    long count = 0;
    for (auto pkgIter = _pPkgCache->PkgBegin(); !pkgIter.end(); ++pkgIter) {
        AptPackage* pPackage =  new AptPackage(pkgIter, _pPkgCache, *_pRecs);
        // TODO: for now remove packages without a description, this removes all non-main-arch packages
        if (!pPackage->isValid()) {
            delete pPackage;
        } else {
            _packageMap[pkgIter.Name()] = pPackage;
            _packages.push_back(pPackage);
        }

        int newProgress = count++ * 100.0 / packageCountEstimate;
        // because of the package count is only an estimate we might exceed 100%
        if (newProgress != progress && progress < 100 && pObserver) {
            progress = newProgress;
            pObserver->setProgress(progress);
        }
    }
    if (pObserver) pObserver->setProgress(100);
}

void AptPackageDB::deletePackages() {
    delete _pCacheFile;
    _pCacheFile = 0;
    delete _pRecs;
    _pRecs = 0;
    for (auto package = _packages.begin(); package != _packages.end(); ++package) {
        delete *package;
    }
    _packages.clear();
    _packageMap.clear();
}

std::list<IPackage*>::const_iterator AptPackageDB::packagesBegin() const {
    return _packages.begin();
}

std::list<IPackage*>::const_iterator AptPackageDB::packagesEnd() const {
    return _packages.end();
}

const IPackage* AptPackageDB::getPackageRecord(const QString &pkg) const {
    auto it = _packageMap.find(pkg.toStdString());
    return (it == _packageMap.end()) ? 0 : it->second;
}

const IPackage* AptPackageDB::getPackageRecord(const string &package) const  {
    auto it = _packageMap.find(package);
    if (it == _packageMap.end()) throw NPlugin::PackageNotFoundException(package);
    return it->second;
}

const QString AptPackageDB::getShortDescription(const string &package) const {
    return getPackageRecord(package)->shortDescription();
}

IPackage::InstalledState AptPackageDB::getState(const string &package) const {
    return getPackageRecord(package)->installedState();
}

void AptPackageDB::reloadPackageInformation(NUtil::IProgressObserver *pObserver) {
    deletePackages();
    initializePackages(pObserver);
}



} // namespace NApt
