//
// C++ Implementation: aptactionplugin
//
// Description: 
//
//
// Author: Benjamin Mesing <bensmail@gmx.net>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include "aptactionplugin.h"

#include <QAction>
#include <QApplication>
#include <QClipboard>

// NApplication
#include "applicationfactory.h"
#include "runcommand.h"

// NPlugin
#include <iprovider.h>

#include "helpers.h"

namespace NPlugin 
{

const QString AptActionPlugin::PLUGIN_NAME = "AptActionPlugin";

AptActionPlugin::AptActionPlugin() :
 	_title("Apt-Action Plugin"),
	_briefDescription("Offers the menu and toolbar entries"),
	_description("This plugin offers the menu and toolbar entries for the APT plugin. "
		"This includes the possibilities to install and remove packages.")
{
	QAction* pQAptUpdateAction = new QAction(tr("Update Apt-Package Database"), this);
	pQAptUpdateAction->setStatusTip(tr("Calls \"apt-get update\" updating the package database"));
	_pAptUpdateAction = new Action(pQAptUpdateAction, false, "System");
	QAction* pQReloadDbAction = new QAction(QObject::tr("Reload Package Database"), this);
	pQReloadDbAction->setStatusTip(tr("Reloads the package database from disk "
		"(e.g. if apt-get update was performed externally)."));
	_pReloadDbAction = new Action(pQReloadDbAction, false, "System");


	QAction* pQAptGetLineAction = new QAction(tr("Copy Command to Install Package to Clipboard"), this);
	pQAptGetLineAction->setToolTip(tr("Creates a command line to install the selected package, and copies it to the clipboard"));
	pQAptGetLineAction->setStatusTip(tr("Creates a command line to install the selected package, and copies it to the clipboard"));
	connect(pQAptGetLineAction, SIGNAL(triggered(bool)), SLOT(onAptGetLineAction()));
	_pAptGetLineAction = new Action(pQAptGetLineAction, true);
	
	QAction* pQAptGetInstallAction = new QAction(tr("Install/Update Package"), this);
	pQAptGetInstallAction->setToolTip(tr("Installs/updates the package"));
	pQAptGetInstallAction->setStatusTip(tr("Installs/updates the package"));
	connect(pQAptGetInstallAction, SIGNAL(triggered(bool)), SLOT(onAptGetInstallAction()));
	_pAptGetInstallAction = new Action(pQAptGetInstallAction, true, "Packages", "Main");

	QAction* pQAptGetRemoveAction = new QAction(tr("Remove Package"), this);
	pQAptGetRemoveAction->setToolTip(tr("Removes the package"));
	pQAptGetRemoveAction->setStatusTip(tr("Removes the package"));
	connect(pQAptGetRemoveAction, SIGNAL(triggered(bool)), SLOT(onAptGetRemoveAction()));
	_pAptGetRemoveAction = new Action(pQAptGetRemoveAction, true, "Packages", "Main");

	QAction* pQSeparatorAction = new QAction(this);
	pQSeparatorAction->setSeparator(true);
	_pSeparatorAction = new Action(pQSeparatorAction, true, "System");
	setInstallationTool(APTITUDE);
}


AptActionPlugin::~AptActionPlugin()
{
	delete _pAptUpdateAction;
	delete _pReloadDbAction;
	delete _pSeparatorAction;
	delete _pAptGetLineAction;
	delete _pAptGetInstallAction;
	delete _pAptGetRemoveAction;
}

/////////////////////////////////////////////////////
// Plugin Interface
/////////////////////////////////////////////////////
	
void AptActionPlugin::init(IProvider* pProvider) 
{
	_pProvider = pProvider; 
	QIcon installIcon(_pProvider->iconDir()+"install-package.png");
	_pAptGetInstallAction->action()->setIcon(installIcon);
	QIcon removeIcon(_pProvider->iconDir()+"remove-package.png");
	_pAptGetRemoveAction->action()->setIcon(removeIcon);
};


/////////////////////////////////////////////////////
// ActionPlugin Interface
/////////////////////////////////////////////////////

vector<Action*> AptActionPlugin::actions() const
{
	vector<Action*> actions;
	actions.push_back(_pSeparatorAction);
	actions.push_back(_pAptUpdateAction);
	actions.push_back(_pReloadDbAction);
	actions.push_back(_pAptGetLineAction);
	actions.push_back(_pAptGetInstallAction);
	actions.push_back(_pAptGetRemoveAction);
	return actions;
}

/////////////////////////////////////////////////////
// Other Methods
/////////////////////////////////////////////////////

void AptActionPlugin::onAptGetLineAction()
{
	QClipboard *pCb = QApplication::clipboard();
	pCb->setText(_installationToolPath + " install "+_pProvider->currentPackage(), QClipboard::Clipboard);
	pCb->setText(_installationToolPath + " install "+_pProvider->currentPackage(), QClipboard::Selection);

}

void AptActionPlugin::onAptGetInstallAction()
{
	aptGetInstallOrRemove(true);
}

void AptActionPlugin::onAptGetRemoveAction()
{
	aptGetInstallOrRemove(false);
}

void  AptActionPlugin::aptGetInstallOrRemove(bool install)
{
	NApplication::ApplicationFactory factory;
	NApplication::RunCommand* pCommand = factory.getRunCommand("");
	pCommand->addArgument(_installationToolPath);
	if (install)
		pCommand->addArgument("install");
	else
		pCommand->addArgument("remove");
	pCommand->addArgument(_pProvider->currentPackage());
	try
	{
		pCommand->startAsRoot();
	}
	catch (const NException::RuntimeException& e)
	{
		_pProvider->reportError(tr("Unable to launch command"), toQString(e.description()));
		delete pCommand;
	}
}

InstallationTool AptActionPlugin::installationTool() 
{
	return _installationTool;
}

void AptActionPlugin::setInstallationTool(InstallationTool tool) 
{
	_installationTool = tool;
	switch (_installationTool) 
	{
		case APT:
			_installationToolPath = "/usr/bin/apt-get";
			break;
		case APTITUDE:
			_installationToolPath = "/usr/bin/aptitude";
			break;
	}
}


}
