#ifndef __BASEPLUGINCONTAINER_H_2004_09_08
#define __BASEPLUGINCONTAINER_H_2004_09_08

#include <qdom.h>

#include <string>
#include <map>


#include "plugincontainer.h"
#include "plugininformer.h"

using namespace std;

namespace NPlugin
{

class IPluginFactory;
class IProvider;

/** Call init() before using the requestPlugin() function. */
class BasePluginContainer : public PluginContainer
{
	typedef map<string, Plugin*> PluginMap;
	/** @brief This maps the plugin names to the corresponding plugins. 
	  * 
	  * If the plugin is not allready loaded it maps to 0.*/
	PluginMap _plugins;
	/** @brief This holds the names of all plugins offered.
	  *
	  * This equals keys(_plugins) and is only used for convenience. */
	vector<string> _pluginNames;
	/** The factory which is used to create the plugins. */
	IPluginFactory* _pFactory;
	/** The provider to be used. */
	IProvider* _pProvider;
protected:
	/** Ths informer used to inform plugin users about changes in the plugins. */
	PluginInformer _pluginInformer;
	/** @brief This adds a new plugin type to the available ones. 
	  *
	  * It should be supported by the plugin factory used. The pointer will 
	  * be set to 0.\n
	  * No plugin with this name must have been added before. 
	  */
	void addPlugin(const string& name);
	/** Returns #_pFactory. */
	IPluginFactory* pluginFactory()	{ return _pFactory; }
	/** Returns #_pProvider. */
	IProvider* provider()	{ return _pProvider; }
public:
	/** @brief Constructs a PluginContainer which uses the given factory to create
	  * the plugins.
	  * 
	  * Call init() before using the requestPlugin() function.
	  */
	BasePluginContainer() {}
	/** @brief This will also delete all plugins owned by this container.
	  *
	  * The plugin users will be informed about the remove.
	  */
	virtual ~BasePluginContainer();
	/** @name PluginContainer Interface
	  *
	  * These functions implement the PluginContainer interface.
	  */
	//@{
	// documented in base class
	virtual Plugin* requestPlugin(const string& name);
	// documented in base class
	virtual void releasePlugin(Plugin* pPlugin);
	// documented in base class
	virtual vector<Plugin*> getLoadedPlugins();
	// documented in base class
	virtual vector<string> offeredPlugins() const	{ return _pluginNames; }
	/** @brief This makes the plugin container ready for use.
	  * 
	  * @param pProvider sets #_pProvider
 	  * @param pFactory sets #_pFactory
	  */
	virtual bool init(IProvider* pProvider, IPluginFactory* pFactory);
	//@}
	/**
	  * 
	  * @param pluginName 
	  */
	QDomElement getSettings(string pluginName) {};
	/**
	  * 
	  * @param pluginName 
	  */
	void isEnabled(string pluginName) {};
	// documented in base class
	virtual void addPluginUser(IPluginUser* pUser)	{ _pluginInformer.addPluginUser(pUser); }
	// documented in base class
	virtual void removePluginUser(IPluginUser* pUser)	{ _pluginInformer.removePluginUser(pUser); };
};

}	// namespace NPlugin

#endif // __BASEPLUGINCONTAINER_H_2004_09_08

