
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: shoutcast.c 2602 2007-07-25 08:32:55Z mschwerin $
 */
#include "config.h"

#include <errno.h>
#include <limits.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"
#include "menu_playback.h"
#include "shoutcast.h"
#include "utils.h"
#include "xmlparser.h"

#ifdef HAVE_SHOUTCAST

extern oxine_t *oxine;

static char *
pls_get_value (const char *buffer, const char *name)
{
    char *line = strstr (buffer, name);
    if (!line)
        return NULL;

    char *start = index (line, '=');
    if (!start)
        return NULL;

    char *end = index (line, '\n');
    if (end)
        end[0] = '\0';

    char *value = ho_strdup (start + 1);

    if (end)
        end[0] = '\n';

    return value;
}


static char *
get_playback_mrl (filelist_t * filelist)
{
    char *mrl = NULL;

    int size = 0;
    char *buffer = read_entire_file (filelist->mrl, &size);

    if (buffer) {
        mrl = pls_get_value (buffer, "File1");
    }

    if (!mrl) {
        error (_("Failed to retrieve %s playback MRL for '%s'!"),
               _("SHOUTcast"), filelist->mrl);
    }

    ho_free (buffer);

    return mrl;
}


static bool
shoutcast_station_name_filter (const char *name)
{
    if (!name)
        return true;

    bool isok = true;
    char *str = conv_lowercase (ho_strdup (name));

    if (strstr (str, "porn")) {
        isok = false;
    }

    ho_free (str);

    return isok;
}


static bool
shoutcast_station_genre_filter (const char *genre)
{
    if (!genre)
        return true;

    bool isok = true;
    char *str = conv_lowercase (ho_strdup (genre));

    if (strstr (str, "porn")) {
        isok = false;
    }

    ho_free (str);

    return isok;
}


static bool
shoutcast_station_rating_filter (const char *rating)
{
    if (!rating) {
        return true;
    }

    int allowed = config_get_number ("streaming.shoutcast.allowed_rating");
    int rated = 10;

    if (strcasecmp (rating, "G") == 0) {
        rated = 0;
    }
    else if (strcasecmp (rating, "PG") == 0) {
        rated = 1;
    }
    else if (strcasecmp (rating, "PG13") == 0) {
        rated = 2;
    }
    else if (strcasecmp (rating, "R") == 0) {
        rated = 3;
    }
    else if (strcasecmp (rating, "NC17") == 0) {
        rated = 4;
    }

    return (allowed >= rated);
}


void
shoutcast_station_read (filelist_t * filelist)
{
    bool success = false;
    char *mrl = get_playback_mrl (filelist);

    if (mrl) {
        filelist_add (filelist, filelist->title, mrl, FILE_TYPE_REGULAR);
        ho_free (mrl);
        success = true;
    }

    if (!success) {
        ho_free (filelist->error);
        filelist->error = ho_strdup (_("Connection failure..."));
    }
}


void
shoutcast_stationlist_read (filelist_t * filelist)
{
    bool success = false;

    int size = 0;
    char *buffer = read_entire_file (filelist->mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    xml_node_t *node;
    xml_parser_init (buffer, size, XML_PARSER_CASE_INSENSITIVE);
    if (xml_parser_build_tree (&node) < 0) {
        error (_("Parsing '%s' failed!"), filelist->mrl);
        goto buffer_free;
    }

    if (strcasecmp (node->name, "stationlist")) {
        error (_("Root node of '%s' must be '%s'!"), filelist->mrl,
               "stationlist");
        goto parser_free;
    }

    /* We assume, that our MRL always starts with http:// */
    char *basemrl = ho_strdup (filelist->mrl);
    if (strncasecmp (basemrl, "http://", 7) != 0) {
        error (_("Unknown prefix to URL '%s'!"), basemrl);
        goto parser_free;
    }
    else {
        char *end = index (basemrl + 8, '/');
        if (end)
            end[0] = '\0';
    }

    /* Scan for tunein */
    const char *tunein = "/sbin/tunein-station.pls";
    xml_node_t *entries = node->child;
    while (entries) {
        if (strcasecmp (entries->name, "tunein") == 0) {
            tunein = xml_parser_get_property (entries, "base");
            break;
        }

        entries = entries->next;
    }

    /* Add all stations */
    entries = node->child;
    while (entries) {
        if (strcasecmp (entries->name, "station") == 0) {
            //shoutcast_station_t *station = ho_new (shoutcast_station_t);

            const char *id = xml_parser_get_property (entries, "id");
            const char *name = xml_parser_get_property (entries, "name");
            const char *genre = xml_parser_get_property (entries, "genre");
            const char *rating = xml_parser_get_property (entries, "rt");

            char *mrl = ho_strdup_printf ("%s/%s?id=%s&filename=playlist.pls",
                                          basemrl, tunein, id);
            char *title = ho_strdup (name);
            if (shoutcast_station_name_filter (name)
                && shoutcast_station_genre_filter (genre)
                && shoutcast_station_rating_filter (rating)) {
                filelist_add (filelist, trim_whitespace (title), mrl,
                              FILE_TYPE_SHOUTCAST_STATION);
            }
            ho_free (mrl);
            ho_free (title);
        }

        entries = entries->next;
    }

    filelist_sort (filelist, NULL);

    success = true;

    ho_free (basemrl);
  parser_free:
    xml_parser_free_tree (node);
  buffer_free:
    ho_free (buffer);

    if (!success) {
        ho_free (filelist->error);
        filelist->error = ho_strdup (_("Connection failure..."));
    }
}

#endif /* HAVE_SHOUTCAST */
