
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playlist_pls.c 2415 2007-07-03 09:15:51Z mschwerin $
 *
 * For information on playlist formats have a look at
 * http://gonze.com/playlists/playlist-format-survey.html
 *
 */
#include "config.h"

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "download.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "playlist.h"
#include "playlist_pls.h"
#include "utils.h"


bool
playlist_pls_save (playlist_t * playlist, const char *mrl)
{
    FILE *f = fopen (mrl, "w");
    if (f == NULL) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return false;
    }

    fprintf (f, "[Playlist]\n");
    fprintf (f, "NumberOfEntries=%d", playlist_length (playlist));

    int i = 1;
    playitem_t *playitem = playlist_first (playlist);
    while (playitem) {
        fprintf (f, "File%d=%s\n", i, playitem->mrl);
        fprintf (f, "Title%d=%s\n", i, playitem->title);
        playitem = playlist_next (playlist, playitem);
        i++;
    }
    fclose (f);

    return true;
}


bool
playlist_pls_load (playlist_t * playlist, const char *mrl)
{
    char *filename = NULL;
    if (is_downloadable (mrl)) {
        filename = download_to_cache (mrl, NULL, true);
    }
    else {
        filename = mrl_get_filename (mrl);
    }

    FILE *f = fopen (filename, "r");
    if (f == NULL) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return false;
    }

    char line[2048];
    bool found_start = false;
    while (fgets (line, 2048, f)) {
        char *cleaned = trim_whitespace (line);
        if (strcasecmp (cleaned, "[Playlist]") == 0) {
            found_start = true;
            break;
        }
    }

    if (!found_start) {
        error (_("The file '%s' is not a valid playlist (PLS)!"), mrl);
        goto out_close;
    }

    int i = 1;
    char *base_dir = get_dirname (mrl);
    char *line_start = ho_strdup_printf ("File%d=", i++);
    while (fgets (line, 2048, f)) {
        char *cleaned = trim_whitespace (line);
        if (starts_with (cleaned, line_start)) {
            char *mrl = ho_strdup (&line[strlen (line_start)]);
            char *title = create_title (mrl);

            /* If the MRL does not start with '/' and it does not contain
             * '://' (which we use to detect URL's e.g. http://, file://)
             * we assume that the file is relative, and prepend the
             * basedir of the playlist file. */
            if ((mrl[0] != '/')
                && (strstr (mrl, "://") == NULL)) {
                char *tmp = ho_strdup_printf ("%s/%s", base_dir, mrl);
                if (file_exists (tmp)) {
                    ho_free (mrl);
                    mrl = tmp;
                }
                else {
                    ho_free (tmp);
                }
            }

            playlist_add (playlist, title, mrl);

            ho_free (title);
            ho_free (mrl);
            ho_free (line_start);
            line_start = ho_strdup_printf ("File%d=", i++);
        }
    }
    ho_free (base_dir);
    ho_free (line_start);
    ho_free (filename);

  out_close:
    fclose (f);
    return found_start;
}
