
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menu_weather.c 2551 2007-07-20 16:59:36Z mschwerin $
 *
 */
#include "config.h"

#include <stdio.h>
#include <unistd.h>

#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "otk.h"
#include "utils.h"
#include "weather.h"

#include "menu_base.h"
#include "menu_main.h"
#include "menu_weather.h"

extern oxine_t *oxine;

#define UPDATE_INTERVAL     (1800)                              /* in seconds */

#ifdef HAVE_WEATHER

static void
update_weather_cb (void *p)
{
    if (!oxine->weather) {
        show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN,
                             "menu_weather.png",
                             _("Retrieving weather data..."));
    }
    else {
        show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN,
                             "menu_weather.png",
                             _("Updating weather data..."));
    }

    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "weather.station_id", &centry);

    /* We put a mutex around weather retrieval, so we don't get into any
     * conflicts with the weather retrieval in the startup_job. */
    mutex_lock (&oxine->weather_mutex);
    weather_free (oxine->weather);
    oxine->weather = get_current_weather (centry.str_value);
    if (oxine->weather) {
        oxine->weather->retrieval_time = time (NULL);
    }
    mutex_unlock (&oxine->weather_mutex);

    if (!oxine->weather) {
        show_message_dialog (update_weather_cb, oxine,
                             show_menu_main, oxine,
                             DIALOG_RETRY_CANCEL,
                             "menu_weather.png",
                             _("Could not retrieve weather data!"));
    }
    else {
        show_menu_weather (oxine);
    }
}


static void
weather_menu_event_handler (void *p, oxine_event_t * event)
{
    if (!is_current_menu (show_menu_weather)) {
        return;
    }
    if (event->type != OXINE_EVENT_KEY) {
        return;
    }

    switch (event->source.key) {
    case OXINE_KEY_BACK:
        show_menu_main (oxine);
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


void
show_menu_weather (void *p)
{
    /* This can only occur if we've just started oxine and the weather has not
     * yet been retrieved. */
    if (!oxine->weather) {
        update_weather_cb (oxine);
        return;
    }

    odk_add_event_handler (oxine->odk, weather_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    create_new_window (false, true);

    char tmp[1024];
    int a1 = OTK_ALIGN_LEFT | OTK_ALIGN_TOP;

    int x1 = 220;
    int x2 = 560;
    int w1 = x2 - x1 - 20;
    int w2 = 780 - x2;
    int y = 110;

    weather_t *weather = oxine->weather;
    snprintf (tmp, 1024, "%s:", _("Weather station"));
    otk_label_new (oxine->otk, x1, y, w1, a1, tmp);
    otk_label_new (oxine->otk, x2, y, w2, a1, weather->station_name);
    y += 40;

    snprintf (tmp, 1024, "%s:", _("Last update"));
    otk_label_new (oxine->otk, x1, y, w1, a1, tmp);
    otk_label_new (oxine->otk, x2, y, w2, a1, weather->update_time);
    y += 40;

    int i = 0;
    for (; i < weather->length; i++) {
        snprintf (tmp, 1024, "%s:", weather->names[i]);
        otk_label_new (oxine->otk, x1, y, w1, a1, tmp);
        otk_label_new (oxine->otk, x2, y, w2, a1, weather->values[i]);
        y += 40;
    }

    otk_widget_t *b;
    b = otk_text_button_new (oxine->otk, 20, 100, 180, 35,
                             _("Update"), update_weather_cb, oxine);
    b = otk_text_button_new (oxine->otk, 20, 140, 180, 35,
                             _("Mainmenu"), show_menu_main, oxine);
    otk_widget_set_focused (b, true);

    set_backto_menu (show_menu_weather, NULL);
    set_current_menu (show_menu_weather, NULL);

    char *background = ho_strdup ("menu_weather.png");
    if (oxine->weather->icon_name) {
        char *mrl = ho_strdup_printf ("%s/backgrounds/%s",
                                      get_dir_oxine_skin (),
                                      oxine->weather->icon_name);
        if (file_exists (mrl)) {
            ho_free (background);
            background = ho_strdup (oxine->weather->icon_name);
        }
        ho_free (mrl);
    }

    show_user_interface (NULL);
    show_menu_background (background);
    ho_free (background);
}

#endif /* HAVE_WEATHER */
