
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id:$
 *
 */
#include "config.h"

#include <stdio.h>
#include <unistd.h>

#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "gui_utils.h"
#include "main_menu.h"
#include "weather_menu.h"
#include "otk.h"
#include "weather.h"

#define UPDATE_INTERVAL     (1800)      // in seconds

#ifdef HAVE_WEATHER

static void
update_weather_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (!oxine->weather)
        show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN,
                             OXINE_BACKGROUNDS "/weathermenu.png",
                             _("Retrieving weather data..."));
    else
        show_message_dialog (NULL, NULL, NULL, NULL, DIALOG_PLAIN,
                             OXINE_BACKGROUNDS "/weathermenu.png",
                             _("Updating weather data..."));

    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "weather.location", &centry);

    /* We put a mutex around weather retrieval, so we don't get into any
     * conflicts with the weather retrieval in the startup_job. */
    mutex_lock (&oxine->weather_mutex);
    weather_free (oxine->weather);
    oxine->weather = get_current_weather (centry.str_value);
    if (oxine->weather)
        oxine->weather->retrieval_time = time (NULL);
    mutex_unlock (&oxine->weather_mutex);

    if (!oxine->weather) {
        show_message_dialog (update_weather_cb, oxine,
                             show_main_menu_cb, oxine,
                             DIALOG_RETRY_CANCEL,
                             OXINE_BACKGROUNDS "/weathermenu.png",
                             _("Could not retrieve weather data!"));
    } else {
        weather_menu_show_cb (oxine);
    }
}


static void
weather_menu_event_handler (void *oxine_p, oxine_event_t * event)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if ((oxine->current_menu == weather_menu_show_cb)
        && (event->type == OXINE_EVENT_KEY)
        && (event->source.key == OXINE_KEY_BACK)) {
        show_main_menu_cb (oxine);
        event->source.key = OXINE_KEY_NULL;
    }
}


void
weather_menu_show_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    /* This can only occur if we've just started oxine and the weather has not
     * yet been retrieved. */
    if (!oxine->weather) {
        update_weather_cb (oxine);
        return;
    }

    mutex_lock (&oxine->weather_mutex);

    hide_user_interface (oxine);

    oxine->current_menu = weather_menu_show_cb;
    oxine->backto_menu = weather_menu_show_cb;

    if (oxine->weather->icon_name) {
        char *image_mrl = ho_strdup_printf ("%s/%s.png",
                                            OXINE_BACKGROUNDS,
                                            oxine->weather->icon_name);
        if (access (image_mrl, R_OK) == 0) {
            show_menu_background (image_mrl);
        } else {
            show_menu_background (OXINE_BACKGROUNDS "/weathermenu.png");
        }
        ho_free (image_mrl);
    } else {
        show_menu_background (OXINE_BACKGROUNDS "/weathermenu.png");
    }

    odk_add_event_handler (oxine->odk, weather_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);

    create_new_window (false, true);

    char tmp[1024];
    int a1 = OTK_ALIGN_LEFT | OTK_ALIGN_TOP;

    int x1 = 220;
    int x2 = 560;
    int w1 = x2 - x1 - 20;
    int w2 = 780 - x2;
    int y = 110;

    weather_t *weather = oxine->weather;
    snprintf (tmp, 1024, "%s:", _("Weather station"));
    otk_label_new (oxine->otk, x1, y, w1, a1, tmp);
    otk_label_new (oxine->otk, x2, y, w2, a1, weather->station_name);
    y += 40;

    snprintf (tmp, 1024, "%s:", _("Last update"));
    otk_label_new (oxine->otk, x1, y, w1, a1, tmp);
    otk_label_new (oxine->otk, x2, y, w2, a1, weather->update_time);
    y += 40;

    int i = 0;
    for (; i < weather->length; i++) {
        snprintf (tmp, 1024, "%s:", weather->names[i]);
        otk_label_new (oxine->otk, x1, y, w1, a1, tmp);
        otk_label_new (oxine->otk, x2, y, w2, a1, weather->values[i]);
        y += 40;
    }

    otk_widget_t *b;
    b = otk_text_button_new (oxine->otk, 20, 100, 180, 35,
                             _("Update"), update_weather_cb, oxine);
    b = otk_text_button_new (oxine->otk, 20, 140, 180, 35,
                             _("Mainmenu"), show_main_menu_cb, oxine);
    otk_widget_set_focus (b);

    show_user_interface (oxine);

    mutex_unlock (&oxine->weather_mutex);
}

#else

void
weather_menu_show_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    show_message_dialog (show_main_menu_cb, oxine, NULL, NULL,
                         DIALOG_OK, OXINE_BACKGROUNDS "/weathermenu.png",
                         _("Both libcurl and libmetar must be\n"
                           "installed for weather support to work!"));
}

#endif /* HAVE_WEATHER */
