#!/usr/bin/env python
# -*- coding: utf-8 -*-

# ows_applet.py
#
# Copyright (C) 2007 David Villa Alises
#
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


# Libwnck Reference Manual
#     http://patg.homeunix.org/doc/libwnck/
#
# Panel Applet Writer's Reference Manual
#     http://developer.gnome.org/doc/API/2.0/panel-applet/libpanel-applet.html
#
# GConf Reference Manual 
#     http://developer.gnome.org/doc/API/2.0/gconf/index.html

import os, sys
import gobject
import gtk, gtk.glade
import gnomeapplet
import bonobo
import wnck
import gconf
import types
#import pyosd
import string
import inspect
import gc
import CORBA
import dbus
import time

import logging as log
log.basicConfig(level=log.DEBUG,
                format='%(asctime)s %(levelname)s %(message)s',
                filename=os.path.join(os.environ['HOME'], '.ows.log'),
                filemode='w')

import traceback, sys
def exception():
    try:
        type, value, tb = sys.exc_info()
        info = traceback.extract_tb(tb)
        filename, lineno, function, text = info[-1] # last line only
        log.error("%s:%d: %s: %s (in %s)" %\
                  (filename, lineno, type.__name__, str(value), function))
    finally:
        type = value = tb = None # clean up

from ows import ows_globals
    
def strcolor(gdkcolor):
    return "#%02X%02X%02X" % (gdkcolor.red  /256,
                              gdkcolor.green/256,
                              gdkcolor.blue /256)

def widget_set_visible(wg, val):
    {False: gtk.Widget.hide, True: gtk.Widget.show_all}[val](wg)

#class xosdOwsOSD(pyosd.osd):
#
#
#    TOP, MID, BOT = pyosd.POS_TOP, pyosd.POS_MID, pyosd.POS_BOT
#    LEFT, CENTER, RIGHT = pyosd.ALIGN_LEFT, pyosd.ALIGN_CENTER, pyosd.ALIGN_RIGHT
#
#    def __init__(self):
#        pyosd.osd.__init__(self, lines=1)
#        self.set_font("-adobe-helvetica-bold-r-normal-*-*-1000-*-*-p-*-*")
#        self.set_colour('red')
#        self.set_align(OwsOSD.RIGHT)
#        self.set_timeout(2)
#        self.set_pos(OwsOSD.TOP)
#        self.set_shadow_offset(4)
#
#        #adaptor
#        self.set_color = self.set_colour
#        self.set_halign = self.set_align
#        self.set_valign = self.set_pos
#        self.set_hoffset = self.set_horizontal_offset
#        self.set_voffset = self.set_vertical_offset
#        self.set_shadow_color = self.set_shadow_colour
#        
#
#osd = pyosd.osd("-adobe-helvetica-bold-r-normal-*-*-1000-*-*-p-*-*",
#                colour="red", timeout=2, align=pyosd.ALIGN_RIGHT, shadow=4,
#                lines=1)


class OwsOSD:

    TOP, MID, BOT = 'top', 'center', 'bottom'
    LEFT, CENTER, RIGHT = 'left', 'center', 'right'

    def __init__(self):
        self.values = {}
        self.xml = string.Template('''
        <message id="ows"
           osd_font="$osd_fontname"
           hide_timeout="$osd_timeout"
           animations="$osd_animation"
           enable_sound="$osd_sound"
           osd_halignment="$osd_halign"
           osd_vposition="$osd_valign"
           drop_shadow="$osd_has_shadow"
           osd_fake_translucent_bg="$osd_translucent"
           avoid_panels="$osd_avoidpanels"
           debug_frame="False"
           ><span foreground="$osd_fontcolor"
           >$msg</span></message>''')

        #self.init_bonobo()
        self.init_dbus()


    def init_bonobo(self):
        osd = bonobo.get_object("OAFIID:GNOME_OSD", "IDL:Bonobo/Application:1.0")
        osd.ref()
        self.osd = bonobo.AppClient(osd)
        self.bus = 'BONOBO'


    def init_dbus(self):
        self.dbus_id = "pt.inescporto.telecom.GnomeOSD"
        self.osd = dbus.SessionBus().get_object(self.dbus_id, "/Server")
        self.bus = 'DBUS'

        
    def display(self, msg):
        self.values['msg'] = msg
        text = self.xml.substitute(self.values)
        if self.bus == 'BONOBO':
            self.osd.msg_send("show-full", [text])
        else:
            self.osd.showMessageFull(text, dbus_interface=self.dbus_id)


    def set_prop(self, name, value):
        self.values[name] = value

    

class OwsPrefs:

    HALIGN = [OwsOSD.LEFT, OwsOSD.CENTER, OwsOSD.RIGHT]
    VALIGN = [OwsOSD.TOP, OwsOSD.MID, OwsOSD.BOT]
    MAX_WS = 36
    RIGHT, LEFT = range(2)

    def __init__(self, applet, osd):
        self.applet = applet
        self.gui = None

        # defaults
        self.sw_num_rows = 1
        self.sw_display_all_workspaces = True
        self.sw_num_workspaces = self.applet.get_ws_count()
        self.sw_display_workspace_name = False
        self.sw_side = self.RIGHT
        
        self.names = ['Workspace %s' % i for i in range(self.MAX_WS)]
        for i in range(self.sw_num_workspaces):
            self.names[i] = self.applet.screen.get_workspace(i).get_name()

        self.colors = [gtk.gdk.color_parse("white")] * self.MAX_WS
        
        self.osd_fontname  = "Arial Bold 48"
        self.osd_single_color = True
        self.osd_fontcolor = "green"
        self.osd_timeout = 1
        self.osd_halign  = 2 #right
        self.osd_valign  = 0 #top
        self.osd_hoffset = 0
        self.osd_voffset = 0
        self.osd_has_shadow = False
        self.osd_shadow_offset = 3
        self.osd_shadow_color  = '#777'
        self.osd_sound = False
        self.osd_animation = False
        self.osd_translucent = False
        self.osd_avoidpanels = False


        self.props = [x[0] for x in inspect.getmembers(self)
                      if x[0].startswith('osd_') or x[0].startswith('sw_')]

        log.debug('props: %s' % self.props)
        
        self.gconf_init()


    def setup_gui(self):
        glade = 'ows/ows.glade'
        if not os.path.exists(glade):
            log.info('cargando el glade del paquete')
            glade = '/usr/share/ows/ows.glade'

        self.gui = gtk.glade.XML(glade, 'dialog')
        self.gui.signal_autoconnect(self)
        self.window = self.gui.get_widget('dialog')
        self.window.connect('delete-event',
                            self.on_button_cancel_clicked)

        # setup combos
        halign_store = gtk.ListStore(str) 
        for i in ['left', 'center', 'right']:
            halign_store.append([i])
            
        self.wg_combo_halign.set_model(halign_store)
        cell = gtk.CellRendererText()
        self.wg_combo_halign.pack_start(cell, True)
        self.wg_combo_halign.add_attribute(cell, 'text', 0)

        valign_store = gtk.ListStore(str) 
        for i in ['top', 'middle', 'bottom']:
            valign_store.append([i])
            
        self.wg_combo_valign.set_model(valign_store)
        cell = gtk.CellRendererText()
        self.wg_combo_valign.pack_start(cell, True)
        self.wg_combo_valign.add_attribute(cell, 'text', 0)

        self.wg_combo_nameside.set_active(0)

        #- setup treeview
        #-  self.model = gtk.ListStore(str, gtk.gdk.Color)
        #-  for i in range(self.applet.get_ws_count()):
        #-      ws = self.applet.screen.get_workspace(i)
        #-      self.model.append([ws.get_name(), gtk.gdk.color_parse("green")])
        #-  
        #-  self.wg_treeview.set_model(self.model)
        #-  self.wg_treeview.set_grid_lines(True)
        #-  
        #-  #self.tvc = gtk.TreeViewColumn('Workspace name')
        #-  #self.wg_treeview.append_column(self.tvc)
        #-  #self.cell = gtk.CellRendererText()
        #-  #self.tvc.pack_start(self.cell, True)
        #-  #self.tvc.add_attribute(self.cell, 'text', 0)
        #-  
        #-  
        #-  tvc1 = gtk.TreeViewColumn("Workscape name", gtk.CellRendererText(), text=0)
        #-  #tvc1.set_resizable(True)
        #-  tvc1.set_expand(True)
        #-  self.wg_treeview.append_column(tvc1)
        #-  
        #-  tvc2 = gtk.TreeViewColumn("color")
        #-  cell = gtk.CellRendererText()
        #-  tvc2.pack_start(cell, True)
        #-  tvc2.add_attribute(cell, 'background-gdk', 1)
        #-  tvc2.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
        #-  #tvc2.set_resizable(True)
        #-  #tvc2.set_max_width(30)
        #-  #tvc2.set_min_width(30)
        #-  tvc2.set_fixed_width(50)
        #-  self.wg_treeview.append_column(tvc2)
        #-  
        #-  self.wg_treeview.show()


        self.wg_spin_num_workspaces.set_range(1, self.MAX_WS)



    def __getattr__(self, name):
        "get widget if attribute starts with 'wg_'"
        if not name.startswith('wg_'):
            raise AttributeError("'%s' isn't a object attribute" % name)
      
        widget = self.gui.get_widget(name[3:])
        if widget == None:
            raise AttributeError("Widget '%s' not found" % name)

        self.__dict__[name] = widget
        return widget        


    def rebuild_workspace_list(self):
        # free old widgets
        for h in self.wg_vbox_ws:
            for c in h:
                h.remove(c)
                del c
            self.wg_vbox_ws.remove(h)
            del h
        gc.collect()

        #print self.colors
        for i in range(self.sw_num_workspaces):
            hbox = gtk.HBox()
            label = gtk.Label(" %2s: " % (i+1))
            entry = gtk.Entry()
            entry.set_text(self.names[i])
            color = gtk.ColorButton(self.colors[i])
            color.set_sensitive(not self.osd_single_color)
            hbox.pack_start(label, expand=False, fill=False)
            hbox.pack_start(entry)
            hbox.pack_start(color, expand=False, fill=False)
            self.wg_vbox_ws.pack_start(hbox, expand=False, fill=False)

        self.wg_vbox_ws.show_all()

    def show(self, *args):
        if self.gui == None:
            self.setup_gui()
            
        self.sw_num_workspaces = self.applet.get_ws_count()
        self.prefs_to_gui()
        self.window.show_all()


    def prefs_to_gui(self):
        print 'PREFS TO GUI--------------'
        
        # Switcher tab
        self.wg_radio_all_ws.set_active(self.sw_display_all_workspaces)
        self.on_radio_all_ws_toggled()
        self.wg_spin_num_rows.set_value(self.sw_num_rows)
        self.wg_spin_num_workspaces.set_value(self.sw_num_workspaces)
        self.wg_check_display_workspace_name.set_active(self.sw_display_workspace_name)
        self.wg_combo_nameside.set_active(self.sw_side)
        self.rebuild_workspace_list()

        # OSD tab
        self.wg_fontbutton.set_font_name(self.osd_fontname)
        print 'single_color', self.osd_single_color
        self.wg_radio_single_color.set_active(self.osd_single_color)
        self.on_radio_single_color_toggled()
        self.wg_fontcolor.set_color(gtk.gdk.color_parse(self.osd_fontcolor))
        self.wg_spin_timeout.set_value(self.osd_timeout)

        self.wg_check_animation.set_active(self.osd_animation)
        self.wg_check_sound.set_active(self.osd_sound)
        self.wg_check_translucent.set_active(self.osd_translucent)
        self.wg_check_avoidpanels.set_active(self.osd_avoidpanels)
        
        self.wg_combo_halign.set_active(self.osd_halign)
        self.wg_combo_valign.set_active(self.osd_valign)
        self.wg_spin_hoffset.set_value(self.osd_hoffset)
        self.wg_spin_voffset.set_value(self.osd_voffset)
        
        self.wg_check_shadow.set_active(self.osd_has_shadow)
        self.wg_spin_shadow_offset.set_value(self.osd_shadow_offset)
        self.wg_shadowcolor.set_color(gtk.gdk.color_parse(self.osd_shadow_color))

    def gui_to_prefs(self):
        # Switcher tab
        self.sw_display_all_workspaces = self.wg_radio_all_ws.get_active()
        self.sw_num_rows = self.wg_spin_num_rows.get_value_as_int()
        self.sw_num_workspaces = self.wg_spin_num_workspaces.get_value_as_int()
        self.sw_display_workspace_name = self.wg_check_display_workspace_name.get_active()
        self.sw_side = self.wg_combo_nameside.get_active()

        # read names and colors
        i = 0
        for h in self.wg_vbox_ws:
            for c in h:
                if isinstance(c, gtk.ColorButton):
                    self.colors[i] = c.get_color()
                if isinstance(c, gtk.Entry):
                    self.names[i] = c.get_text()
            i += 1


        # OSD tab
        self.osd_fontname      = self.wg_fontbutton.get_font_name()
        self.osd_single_color  = self.wg_radio_single_color.get_active()
        self.osd_fontcolor     = strcolor(self.wg_fontcolor.get_color())
        self.osd_timeout       = self.wg_spin_timeout.get_value()

        self.osd_animation     = self.wg_check_animation.get_active()
        self.osd_sound         = self.wg_check_sound.get_active()
        self.osd_translucent   = self.wg_check_translucent.get_active()
        self.osd_avoidpanels   = self.wg_check_avoidpanels.get_active()

        self.osd_halign        = self.wg_combo_halign.get_active()
        self.osd_valign        = self.wg_combo_valign.get_active()
        self.osd_hoffset       = self.wg_spin_hoffset.get_value_as_int()
        self.osd_voffset       = self.wg_spin_voffset.get_value_as_int()

        self.osd_has_shadow    = self.wg_check_shadow.get_active()
        self.osd_shadow_offset = self.wg_spin_shadow_offset.get_value_as_int()
        self.osd_shadow_color  = strcolor(self.wg_shadowcolor.get_color())


    def on_spin_num_workspaces_value_changed(self, wg):
        self.sw_num_workspaces = wg.get_value_as_int()
        self.rebuild_workspace_list()

    def on_radio_all_ws_toggled(self, *args):
        self.wg_spin_num_rows.set_sensitive(self.wg_radio_all_ws.get_active())

    def on_check_display_workspace_name_toggled(self, wg):
        self.wg_combo_nameside.set_sensitive(wg.get_active())
        

    def on_radio_single_color_toggled(self, *args):
        self.wg_fontcolor.set_sensitive(self.wg_radio_single_color.get_active())
        self.osd_single_color = self.wg_radio_single_color.get_active()
        self.rebuild_workspace_list()

    def on_check_shadow_toggled(self, wg):
        self.wg_hbox_shadow.set_sensitive(self.wg_check_shadow.get_active())

    def on_button_apply_clicked(self, button):
        self.gui_to_prefs()
        self.applet.reconfigure()

    def on_button_cancel_clicked(self, button, *args):
        self.gconf_to_prefs()
        self.window.hide()
        return True

    def on_button_ok_clicked(self, button):
        self.gui_to_prefs()
        self.prefs_to_gconf()
        self.window.hide()
                
        #self.applet.show_workspace_name()
        self.applet.reconfigure()


    ##- GConf stuff
    def gconf_init(self):
        self.client = gconf.client_get_default()
        
        #self.applet_key = self.applet.get_preferences_key()
        #log.debug("preferences_key: %s" % self.applet_key)
        #if self.applet_key == None:
        #     self.applet_key = "/apps/panel/applets/applet_23/prefs"
        self.applet_key = '/apps/panel/applets/ows_screen0/prefs/'

        self.applet.add_preferences('/schemas/app/ows')
        self.client.add_dir(self.applet_key[:1], gconf.CLIENT_PRELOAD_NONE)

        self.gconf_to_prefs()


    def gconf_to_prefs(self):
        'Load preferences'
        casts = {gconf.VALUE_BOOL:   gconf.Value.get_bool,
                 gconf.VALUE_INT:    gconf.Value.get_int,
                 gconf.VALUE_FLOAT:  gconf.Value.get_float,
                 gconf.VALUE_STRING: gconf.Value.get_string}
        
        for name in self.props:
            gval = self.client.get(self.applet_key + name)
            if gval == None: continue
            setattr(self, name, casts[gval.type](gval))
            log.debug("load %s: %s" % (name, casts[gval.type](gval)))

        aux = [gtk.gdk.color_parse(x) for x in
               self.client.get_list(self.applet_key + 'colors',
                                   gconf.VALUE_STRING)]

        if aux: self.colors = aux
        

    def prefs_to_gconf(self):
        'Save preferences'
        casts = {types.BooleanType: gconf.Client.set_bool,
                 types.IntType:     gconf.Client.set_int,
                 types.FloatType:   gconf.Client.set_float,
                 types.StringType:  gconf.Client.set_string}
        
        for name in self.props:
            value = getattr(self,name)
            casts[type(value)](self.client,
                               self.applet_key + name, value)
            log.debug("save %s: %s" % (name, value))

        self.client.set_list(self.applet_key + 'colors', gconf.VALUE_STRING,
                             [strcolor(x) for x in self.colors])
        


class Pager(wnck.Pager):
    CONTENT, NAMES = range(2)

    def __init__(self, applet, screen):
        wnck.Pager.__init__(self, screen)
        self.applet = applet
        self.screen = screen
        self.__n_rows = 1
        self.__show_all = True
        #self.set_shadow_type(gtk.SHADOW_IN)
        self.mode = None

    def set_n_rows(self, n):
        self.__n_rows = n
        wnck.Pager.set_n_rows(self, n)

    def set_show_all(self, val):
        self.__show_all = val
        wnck.Pager.set_show_all(self, val)

    def adjust_size(self, pager_h=None):
        if pager_h == None:
            pager_h = self.applet.get_size()
        
        ratio = float(self.screen.get_width()) / self.screen.get_height()
        #print ratio
        #print self.__n_rows

        if not self.__show_all:
            self.set_size_request(int(pager_h * ratio), pager_h)
            return

        n_ws = self.applet.get_ws_count()
        pager_w = int(pager_h * ratio * n_ws / (self.__n_rows ** 2))
        print pager_w, pager_h

        self.set_size_request(pager_w, pager_h)
        

class OwsApplet(gnomeapplet.Applet):

    menu = '''
<popup name="button3">
   <menuitem name="Item 2" verb="Preferences" label="Preferences"
      pixtype="stock" pixname="gtk-preferences"/>
   <menuitem name="Item 1" verb="About" label="About..."
      pixtype="stock" pixname="gtk-about"/>
</popup>
'''

    def __init__(self):
        gnomeapplet.Applet.__init__(self)
        self.workspace = None
        self._name_change_handler_id = None
        self.source_timer = 7


    def applet_init(self):
	self.screen = wnck.screen_get_default()

        self.osd = OwsOSD()
        self.pager = Pager(self, self.screen)
        self.prefs = OwsPrefs(self, self.osd)
        
        self.setup_gui()
        self.reconfigure()

        self.window_pager = None
       
        self.setup_menu(self.menu, [("Preferences", self.prefs.show), 
                                    ("About", self.about_cb)])

        # applet signales
        self.connect("change_background", self.background_changed)
        self.connect("change_orient",     self.orient_changed) 

        # screen signals
	self.screen.connect("active_workspace_changed",
                            self.on_workspace_changed)

        self.screen.connect("workspace_created",   self.on_layout_changed)
        self.screen.connect("workspace_destroyed", self.on_layout_changed)

        # workspace signals
        for i in range(self.get_ws_count()):
            ws = self.screen.get_workspace(i)
            ws.connect('name-changed', self.show_workspace_name)
            

    def setup_gui(self):
        self.button = gtk.Button()
        self.button.set_relief(gtk.RELIEF_NONE)
        self.button.connect('clicked', self.on_button_clicked)
        self.label = gtk.Label()
        self.label.set_use_markup(True)
        self.label.set_padding(0, 0)
        self.button.add(self.label)
        print 'bw', self.button.get_border_width()
        
        self.box = None
        self.external_pager = None
        self.orient_changed()


    def setup_extPager(self):
        self.window_pager = gtk.Window()
        self.window_pager.set_title('')
        self.window_pager.set_type_hint(gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
        self.window_pager.set_position(gtk.WIN_POS_MOUSE)
        self.window_pager.set_keep_above(True)
        self.window_pager.set_property('resizable', False)
        self.window_pager.set_property("skip-pager-hint", True)
        self.window_pager.set_property("skip-taskbar-hint", True)
        
        self.external_pager = Pager(self, self.screen)
        self.window_pager.add(self.external_pager)
        self.window_pager.connect('delete-event', self.on_window_pager_delete_event)


    def reconfigure(self):
        if self.prefs.sw_display_workspace_name:
            self.button.set_no_show_all(False)
            self.button.show()
        else:
            self.button.set_no_show_all(True)
            self.button.hide()

        self.orient_changed()

        # pager
        self.pager.set_n_rows(self.prefs.sw_num_rows)
        self.pager.set_show_all(self.prefs.sw_display_all_workspaces)
        self.pager.adjust_size()

        # OSD
        for name in self.prefs.props:
            self.osd.set_prop(name, getattr(self.prefs, name))
            
        self.osd.set_prop('osd_timeout', int(self.prefs.osd_timeout*1000))
        self.osd.set_prop('osd_halign',  OwsPrefs.HALIGN[self.prefs.osd_halign])
        self.osd.set_prop('osd_valign',  OwsPrefs.VALIGN[self.prefs.osd_valign])

        # Screen/workspaces
        self.screen.change_workspace_count(self.prefs.sw_num_workspaces)
        for i in range(self.prefs.sw_num_workspaces):
            while gtk.events_pending(): gtk.main_iteration()
            self.screen.get_workspace(i).change_name(self.prefs.names[i])

        #while gtk.events_pending(): gtk.main_iteration()
        self.show_workspace_name(self.screen.get_active_workspace())
        #self.on_workspace_changed()

    def get_ws_count(self):
        while gtk.events_pending(): gtk.main_iteration()
        return self.screen.get_workspace_count()
    

    def on_layout_changed(self, *args):
        # FIXME: Hay que controlar la creación y destrucción de
        # workspaces para conectar/desconectar la señal name-changed,
        # tal como está ahora la señal name-changed no funcionará para
        # workspace creados después de añadir el applet.
        log.debug('workspaces: %s' % self.screen.get_workspace_count())
        self.pager.adjust_size()
        

    def on_button_clicked(self, button):
        'Show external pager'        
        if self.window_pager == None:
            self.setup_extPager()

        print self.external_pager.mode

        if self.external_pager.mode == Pager.NAMES:
            self.window_pager.hide()
            self.external_pager.mode = None
            return

        if self.external_pager.mode == Pager.CONTENT:
            self.external_pager.set_display_mode(wnck.PAGER_DISPLAY_NAME)
            self.external_pager.mode = Pager.NAMES

        else:
            self.external_pager.set_display_mode(wnck.PAGER_DISPLAY_CONTENT)
            self.external_pager.set_n_rows(self.prefs.sw_num_rows)
            self.external_pager.mode = Pager.CONTENT

        self.external_pager.adjust_size(96)
        self.window_pager.show_all()
        
        #if button.get_active():
        #    self.external_pager.set_n_rows(self.prefs.sw_num_rows)
        #    self.window_pager.show_all()
        #else:
        #    self.window_pager.hide()


    def on_window_pager_delete_event(self, win, *args):
        self.external_pager.mode = None
        win.hide()
        return True

    def on_workspace_changed(self, screen=None, ws=None):
        while gtk.events_pending(): gtk.main_iteration()
        ws = screen.get_active_workspace()

        if self.external_pager:
            self.window_pager.hide()
            self.external_pager.mode = None

        self.show_workspace_name(ws)
        #if self.source_timer:
        #    gtk.timeout_remove(self.source_timer)
        #
        #self.source_timer = gtk.timeout_add(350, self.show_workspace_name, ws)


    def show_workspace_name(self, ws=None):
        
        if not self.prefs.osd_single_color:
            self.osd.set_prop('osd_fontcolor',
                              strcolor(self.prefs.colors[ws.get_number()]))
        
        self.osd.display(ws.get_name())
	self.label.set_markup('<span size="small">%s</span>' % ws.get_name())
	self.show_all()
        return False


#    def on_workspace_changed(self, screen=None):
#        while gtk.events_pending(): gtk.main_iteration()
#        #self.active_ws = self.screen.get_active_workspace()
#
#        if self.external_pager:
#            self.window_pager.hide()
#            self.external_pager.mode = None
#
#        ws = screen.get_active_workspace()
#
#        log.debug('changed')
#        log.debug(str(ws))
#
#	self.show_workspace_name(ws)
#        return
#
#        if self.source_timer:
#            gobject.source_remove(self.source_timer)
#
#        self.source_timer = gobject.timeout_add(350, self.show_workspace_name, ws)



    def background_changed(self, applet, bgtype, color, pixmap):
        for w in [self, self.pager, self.button]:
            w.modify_bg(gtk.STATE_NORMAL, color)
            if pixmap is not None:
                w.get_style().bg_pixmap[gtk.STATE_NORMAL] = pixmap

    def orient_changed(self, *args):
        angles = {gnomeapplet.ORIENT_LEFT:  270,
                  gnomeapplet.ORIENT_RIGHT:  90}

        if self.box:
            self.box.remove(self.pager)
            self.box.remove(self.button)
            self.remove(self.box)
            del self.box

        orient = self.get_orient()
        if orient in angles.keys():
            self.box = gtk.VBox()
            self.pager.set_orientation(gtk.ORIENTATION_VERTICAL)
        else:
            self.box = gtk.HBox()
            self.pager.set_orientation(gtk.ORIENTATION_HORIZONTAL)

        if self.prefs.sw_side == OwsPrefs.RIGHT:
            pack_func = self.box.pack_start
        else:
            pack_func = self.box.pack_end            

        pack_func(self.pager)
        pack_func(self.button)
        self.add(self.box)
        self.show_all()
        
        self.label.set_angle(angles.get(orient, 0))

    def about_cb(self, event, data=None):
        about = gtk.AboutDialog()
        about.set_name(ows_globals.long_name)
        about.set_version(ows_globals.version)
        about.set_authors(ows_globals.authors)
        #about.set_artists(ows_globals.artists)
        #about.set_translator_credits(_('translator-credits'))
        about.set_logo(gtk.gdk.pixbuf_new_from_file(ows_globals.image_dir + "/ows.png"))
        about.set_license(ows_globals.license)
        about.set_wrap_license(True)
        about.set_copyright(ows_globals.copyright)
        
        #gtk.about_dialog_set_url_hook(open_site, ows_globals.website)
        about.set_website(ows_globals.website)
        about.run()
        about.destroy()


def ows_factory_init(applet, id=None):
    return applet.applet_init()


#To test in a standalone window, type "./ows_applet.py window"
if len(sys.argv) == 2 and sys.argv[1] == "window":

    console = log.StreamHandler()
    console.setLevel(log.DEBUG)
    console.setFormatter(log.Formatter('%(levelname)-8s: %(message)s'))
    log.getLogger().addHandler(console)
    
    main_window = gtk.Window(gtk.WINDOW_TOPLEVEL)
    main_window.connect("destroy", gtk.main_quit)
    app = OwsApplet()
    app.reparent(main_window)
    ows_factory_init(app)
    main_window.show_all()
    gtk.main()
    sys.exit()


log.debug('-- applet init')
try:
    gobject.type_register(OwsApplet)
    gnomeapplet.bonobo_factory("OAFIID:GNOME_OSDWorkspaceSwitcher_Factory", 
                               OwsApplet.__gtype__, 
                               "workspace OSD", "0", ows_factory_init)
except Exception, e:
    exception()
    
