#!/bin/sh

# Config files can define several variables used throughout this script.
# JAVACMD - The java command to launch osmosis.
# JAVACMD_OPTIONS - The options to append to the java command, typically used to modify jvm settings such as max memory.
# OSMOSIS_OPTIONS - The options to apply to all osmosis invocations, typically used to add plugins or make quiet operation the default.

if [ -f /etc/osmosis/osmosis.conf ] ; then
  # shellcheck source=/dev/null
  . /etc/osmosis/osmosis.conf
fi

if [ -f "$HOME/.osmosis" ] ; then
  # shellcheck source=/dev/null
  . "$HOME/.osmosis"
fi

if [ -z "$JAVACMD" ] ; then
  # No JAVACMD provided in osmosis config files, therefore default to java
  JAVACMD=java
fi

## resolve links - $0 may be a link to application
PRG="$0"

# if started without absolute path, but from PATH environment
if [ ! -s "$PRG" ] ; then
  PRG=$(which "$PRG")
fi

# need this for relative symlinks
while [ -h "$PRG" ] ; do
  ls=$(ls -ld "$PRG")
  link=$(expr "$ls" : '.*-> \(.*\)$')
  if expr "$link" : '/.*' > /dev/null; then
    PRG="$link"
  else
    PRG="$(dirname "$PRG")/$link"
  fi
done

if [ "$1" = "" ] || echo "$@" | grep -q -e '--help' ; then
cat <<EOF
osmosis

Example Usage

Import a planet file into a local PostgreSQL database.

osmosis --read-xml file=~/osm/planbet/planet.osm --write-apidb host="x" database="x" user="x" password="x"

Export a planet file from a local PostgreSQL database.

osmosis --read-apidb host="x" database="x" user="x" password="x" --write-xml file="planet.osm"

Derive a change set between two planet files.

osmosis --read-xml file="planet2.osm" --read-xml file="planet1.osm" --derive-change --write-xml-change file="planetdiff-1-2.osc"

Derive a change set between a planet file and a database.

osmosis --read-mysql host="x" database="x" user="x" password="x" --read-xml file="planet1.osm" --derive-change --write-xml-change file="planetdiff-1-2.osc"

Apply a change set to a planet file.

osmosis --read-xml-change file="planetdiff-1-2.osc" --read-xml file="planet1.osm" --apply-change --write-xml file="planet2.osm"

Sort the contents of a planet file.

osmosis --read-xml file="data.osm" --sort type="TypeThenId" --write-xml file="data-sorted.osm"

The above examples make use of the default pipe connection feature, however a simple read and write planet file command line could be written in two ways. The first example uses default pipe connection, the second explicitly connects the two components using a pipe named "mypipe". The default pipe connection will always work so long as each task is specified in the correct order.

osmosis --read-xml file="planetin.osm" --write-xml file="planetout.osm"

osmosis --read-xml file="planetin.osm" outPipe.0="mypipe" --write-xml file="planetout.osm" inPipe.0="mypipe"

Full usage details are available at: http://wiki.openstreetmap.org/wiki/Osmosis/Detailed_Usage

EOF
exit 1
fi

# make it fully qualified
MYAPP_HOME="/usr/share/osmosis"
LIBRARIES_HOME="/usr/share/java"

# Build up the classpath of required jar files via classworlds launcher.
MYAPP_CLASSPATH=$LIBRARIES_HOME/plexus-classworlds.jar

MAINCLASS=org.codehaus.classworlds.Launcher
EXEC="$JAVACMD $JAVACMD_OPTIONS -cp $MYAPP_CLASSPATH"
EXEC="$EXEC -Dapp.home=$MYAPP_HOME"
EXEC="$EXEC -Dclassworlds.conf=/etc/osmosis/plexus.conf"
EXEC="$EXEC -Dlog4j.configuration=file:/etc/osmosis/log4j.properties"
EXEC="$EXEC $MAINCLASS $OSMOSIS_OPTIONS"

exec $EXEC "$@"
