// $Id: FilePathContainer.cpp,v 1.2 2004/04/20 12:26:10 andersb Exp $
/**
 * OsgAL - OpenSceneGraph Audio Library
 * Copyright (C) 2004 VRlab, Ume University
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
 */


#include "osgAL/FilePathContainer"
#include <sys/stat.h>

using namespace osgAL;

FilePathContainer::FilePathContainer()
{
  // Set path delimiter for environment variable depending on operative system
  m_delimiter = FP_PATH_DELIMITER;
}

FilePathContainer::FilePathContainer(const std::string& env)
{
  m_delimiter = FP_PATH_DELIMITER;

  setEnvironmentVariable(env);

}


FilePathContainer::~FilePathContainer()
{ 
  m_path_list.clear(); 
}


void FilePathContainer::setFilePathList(const std::string& paths, const std::string& delimitor) 
{ 
  m_path_list.clear(); 
  convertStringToList(paths, m_path_list, delimitor); 
}

std::list<std::string>& FilePathContainer::getFilePathList() 
{ 
  return m_path_list; 
}


bool FilePathContainer::setEnvironmentVariable(const std::string& env)
{

  // Clear path list
  m_path_list.clear();

  const char *path;
  if( (path = getenv(env.c_str()))){

    // Set path list to paths existing in environment variable
    convertStringToList(path, m_path_list, m_delimiter);

    return true;
  }

  return false;
}


void FilePathContainer::setFilePath(const std::string& path)
{

  // Clear path list
  m_path_list.clear();

  // Set path list to paths existing in environment variable
  convertStringToList(path, m_path_list, m_delimiter);

}

void FilePathContainer::addFilePath(const std::string& path)
{

  PathList pl;
  // Set path list to paths existing in environment variable
  convertStringToList(path, pl, m_delimiter);

  // Merge the new with the previous one
  m_path_list.merge(pl);
}



void FilePathContainer::convertStringToList(const std::string& str, std::list<std::string>& lst, const std::string& delimiter)
{

  std::string::size_type start = 0, end;
  std::string sub_str;

  do {

    end = str.find(m_delimiter, start);

    sub_str = str.substr(start, end - start);

    lst.push_back(sub_str);

    start = end + delimiter.size();

  } while (end != std::string::npos);


}


/*----------------------------------------------------------------------------
 Extract directory from path (remove filename)
 ----------------------------------------------------------------------------*/
std::string FilePathContainer::extractDirectory(const std::string& path)
{
  std::string::size_type slash = path.find_last_of('/');

  if (slash == std::string::npos) 
    return std::string("");

  return std::string(path.begin(), path.begin() + slash + 1);
  
}


/*----------------------------------------------------------------------------
 Extract filename from path
 ----------------------------------------------------------------------------*/
std::string FilePathContainer::extractFilename(const std::string& path)
{

  std::string::size_type slash = path.find_last_of('/');

  if (slash == std::string::npos) 
    return path;

  return std::string(path.begin() + slash + 1, path.end());

}


std::string FilePathContainer::findFile(const std::string& filename)
{

  std::list<std::string>::iterator list_it;

  // Loop through all paths and try to find file
  for (list_it = m_path_list.begin(); list_it != m_path_list.end(); list_it++) {

    std::string full_path = (*list_it);

    // Make sure path ends with "/"
    if ((!full_path.empty()) && (full_path[full_path.size() - 1] != '/') && (full_path[full_path.size() - 1] != '\\'))
      full_path.push_back('/');

    // Build path to file
    full_path += filename;
    
    // Debug
    //std::cerr << "Searching path: " << full_path << std::endl;

    /* Ok assume filename is a directory, assume further if
       someone have specified the directory with a / or a \ at the end,
       stat cant find that directory/file. 
       If we remove any existing /\, what will that mean?
       We will find a directory, but if someone by mistake adds a  \/ to
       a path, that error wont be discovered due to this.
       Just something to think about...
    */
    if (full_path.length()) {
      char s = full_path[full_path.length()-1];
      if (s == '\\' || s == '/')
        full_path = full_path.substr(0, full_path.length()-1);
    }

    // Can we access the file with this path?
    struct stat buf;
    // Does the file exist and we can read it?
    if (!stat( full_path.c_str(), &buf )) {
        if (buf.st_mode & S_IREAD ) {
          return std::string(full_path.begin(), full_path.end());
        }
    }

  }
  
  // Return empty string if file wasn't found
  return std::string("");
}

void FilePathContainer::pushbackPath(const std::string& path)
{
  m_path_list.push_back(path);
}

void FilePathContainer::pushfrontPath(const std::string& path)
{
  m_path_list.push_front(path);
}



/*------------------------------------------

* $Source: /afs/hpc2n.umu.se/projects/vrutils/Repository//osgAL/src/osgAL/FilePathContainer.cpp,v $
* $Revision: 1.2 $ 
* $Date: 2004/04/20 12:26:10 $
* $Author: andersb $ 
* $Locker:  $

* Description: 
 
 Copyright (C) Peter Sunna, VRlab, Ume University 2002

 Created: 2002-06-26

* $Log: FilePathContainer.cpp,v $
* Revision 1.2  2004/04/20 12:26:10  andersb
* Added SoundRoot
*
* Revision 1.1  2004/03/03 07:50:52  andersb
* *** empty log message ***
*

----------------------------------------------------------------------------*/

