
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <debug/memory.h>
#include <debug/log.h>

#include "images.h"
#include "router.h"
#include "mkdirhier.h"
#include "db.h"
#include "html.h"

#define NB_OF(x) (sizeof (x) / sizeof ((x)[0]))

static char errors[4096];

static const char *html_get_error ()
{
   return (errors);
}

static const char *html_set_error (const char *fmt, ...)
{
   va_list ap;
   va_start (ap,fmt);
   vsnprintf (errors,NB_OF(errors),fmt,ap);
   va_end (ap);
   errors[NB_OF(errors) - 1] = '\0';
   return (errors);
}

#define DIM_IO		0
#define DIM_ALERTS	1
#define DIM_STATUS	2

static const char *wstr (const char *filename,int fd,const char *str)
{
   size_t len = strlen (str);
   ssize_t result = write (fd,str,len);

   if (result < 0)
	 return (html_set_error ("write error while writing to %s: %s",filename,strerror (errno)));

   if (result != len)
	 return (html_set_error ("short write count while writing to %s: wrote %d bytes, expected %u bytes",filename,result,len));

   return (NULL);
}

static const char *gen_html (const char *filename,const char *root,int *width,int *height,const router_t *router,int n,int rc,int ic)
{
   static const char *pieces[6] =
	 {
		"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n"
		"<html LANG=\"en\">\n"
		"<head>\n"
		"<title>Orca Router Monitoring</title>\n"
		"<meta http-equiv=\"Content-Language\" content=\"en\">\n"
		"<meta http-equiv=\"Expires\" content=\"0\">\n"
		"<meta http-equiv=\"Pragma\" content=\"no-cache\">\n"
		"<meta http-equiv=\"Cache-Control\" content=\"no-cache, must-revalidate\">\n"
		"<meta http-equiv=\"Content-Style-Type\" content=\"text/css\">\n"
		"<style type=\"text/css\">\n"
		"<!--\n"
		"BODY {background: white; color: black; font-family: Arial, Helvetica, sans-serif;\n"
		"margin-top: 0; margin-left: 0; margin-right: 0;}\n"
		"A.MAINNAV:link {font-family: Arial,Helvetica,sans-serif; font-weight: bold;\n"
		"text-decoration: underline; color: #000000;}\n"
		"A.MAINNAV:visited {font-family: Arial,Helvetica,sans-serif; font-weight: bold;\n"
		"text-decoration: underline; color: #000000;}\n"
		"A.MAINNAV:hover {font-family: Arial,Helvetica,sans-serif; font-weight: bold;\n"
		"text-decoration: underline; color: #999999;}\n"
		"A.SUBNAV:link {font-family: Arial,Helvetica,sans-serif; font-weight: bold;\n"
		"text-decoration: underline; color: #ffffff;}\n"
		"A.SUBNAV:visited {font-family: Arial,Helvetica,sans-serif; font-weight: bold;\n"
		"text-decoration: underline; color: #ffffff;}\n"
		"A.SUBNAV:hover {font-family: Arial,Helvetica,sans-serif; font-weight: bold;\n"
		"text-decoration: underline; color: #999999;}\n"
		"H1 {font-family: Arial,Helvetica,sans-serif; color: #999999;}\n"
		"TD {font-family: Arial,Helvetica,sans-serif; font-weight: bold;}\n"
		"-->\n"
		"</style>\n"
		"</head>\n"
		"<body bgcolor=\"#ffffff\" text=\"#000000\" link=\"#6666cc\" vlink=\"#6666cc\" alink=\"#ff6600\" topmargin=\"0\" leftmargin=\"0\">\n"
		"<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" width=\"100%%\">\n"
		"<tr align=\"center\">\n"
		"<td valign=\"center\" width=\"200\" height=\"150\" colspan=\"2\" rowspan=\"2\" bgcolor=\"#000000\" nowrap>\n",
		  /* image url */
		"</td>\n"
		"<td valign=\"top\" height=\"75\" bgcolor=\"#000000\" nowrap>\n"
		"<table bgcolor=\"#000000\" border=\"0\" cellspacing=\"0\" cellpadding=\"0\" align=\"left\">\n"
		"<tr align=\"center\">\n",
		  /* interfaces */
		"</tr>\n"
		"</table>\n"
		"</td>\n"
		"</tr>\n",
		  /* traffic analysis for ... */
		"<tr>\n"
		"<td width=\"15\" bgcolor=\"#6666cc\" nowrap>&nbsp;</td>\n"
		"<td valign=\"top\" bgcolor=\"#6666cc\" nowrap>\n"
		"<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" align=\"left\">\n",
		  /* routers */
		"</table>\n"
		"</td>\n"
		"<td>\n"
		"<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" width=\"100%%\">\n"
		"<tr><td rowspan=\"10\" width=\"100\" nowrap>&nbsp;</td><td>&nbsp;</td></tr>\n",
		  /* pictures */
		"</table>\n"
		"</td>\n"
		"</tr>\n"
		"</table>\n"
		"</body>\n"
		"</html>\n"
	 };
   int i,fd,sep = 0;
   const char *msg;
   interface_t *iface,*cur = NULL;

   if ((fd = creat (filename,0644)) < 0)
	 return (html_set_error ("couldn't create file %s: %s",filename,strerror (errno)));

#define PIECE(x) do {									\
   if ((msg = wstr (filename,fd,pieces[x])) != NULL)	\
	 {													\
		close (fd);										\
		return (msg);									\
	 }													\
} while (0)

   /*
	* we don't check the returned size (difficult) of dprintf()'s, but
	* hopefully the write()'s later on will catch the error
	*/

   PIECE(0);

   dprintf (fd,"<img src=\"%s/images/orca.png\" width=\"158\" height=\"62\" alt=\"orca\">\n",root);

   PIECE(1);

   for (iface = router[rc].iface, i = 0; iface != NULL; iface = iface->next)
	 if (iface->status[IF_ADMIN_STATUS] == IF_STATUS_UP || !router[rc].ignore_admin_down)
	   {
		  /* if we're not next to the current interface, put a seperator */
		  if (i && (sep > 1 || (!sep && iface->idx != ic)))
			dprintf (fd,"<td height=\"37\" nowrap><img src=\"%s/images/seperator.png\" width=\"8\" height=\"27\" alt=\"seperator\"></td>\n",root);

		  /* if we just had the current interface, incidate that seperators should follow */
		  if (sep == 1) sep++;

		  if (iface->idx == ic)
			{
			   /* current interface */
			   sep = 1;
			   dprintf (fd,
						"<td height=\"37\" bgcolor=\"#6666cc\" nowrap>&nbsp;&nbsp;<font color=\"#000000\">%s</font>&nbsp;&nbsp;</td>\n",
						iface->descr);
			   cur = iface;
			}
		  else
			{
			   /* link to another interface */
			   dprintf (fd,
						"<td height=\"37\" nowrap>&nbsp;<a href=\"%s/%s/%d/\" class=\"SUBNAV\">&nbsp;%s</a>&nbsp;</td>\n",
						root,router[rc].hostname,iface->idx,iface->descr);
			}

		  i++;
	   }

   /* unless we just had the current interface, add a seperator to the end */
   if (sep != 1) dprintf (fd,"<td height=\"37\" nowrap><img src=\"%s/images/seperator.png\" width=\"8\" height=\"27\" alt=\"seperator\"></td>\n",root);

   PIECE(2);

   if (cur->alias != NULL)
	 dprintf (fd,"<tr><td colspan=\"2\" nowrap><h1>&nbsp;&nbsp;&nbsp;%s</h1></td></tr>\n",cur->alias);
   else
	 dprintf (fd,"<tr><td colspan=\"2\" nowrap><h1>&nbsp;&nbsp;&nbsp;Traffic Analysis for %s</h1></td></tr>\n",cur->descr);

   PIECE(3);

   for (i = 0; i < n; i++)
	 {
		dprintf (fd,"<tr><td>&nbsp;</td></tr>\n");
		if (i == rc)
		  dprintf (fd,"<tr><td><font color=\"#ffffff\">%s</td></tr>\n",router[i].name);
		else
		  {
			 for (iface = router[i].iface; iface != NULL; iface = iface->next)
			   if (iface->status[IF_ADMIN_STATUS] == IF_STATUS_UP || !router[i].ignore_admin_down)
				 break;

			 if (iface != NULL)
			   dprintf (fd,"<tr><td><a href=\"%s/%s/%d/\" class=\"MAINNAV\">%s</a></td></tr>\n",root,router[i].hostname,iface->idx,router[i].name);
			 else
			   log_printf (LOG_WARNING,"router %s have no configured interfaces!\n",router[i].hostname);
		  }
	 }

   PIECE(4);

   dprintf (fd,
			"<tr><td>Data sent / received</td></tr>\n"
			"<tr><td nowrap><img src=\"%s/%s/%d/io.png\" width=\"%d\" height=\"%d\" alt=\"i/o\"></td></tr>\n"
			"<tr><td>&nbsp;</td></tr>\n"
			"<tr><td>Alerts</td></tr>\n"
			"<tr><td nowrap><img src=\"%s/%s/%d/alerts.png\" width=\"%d\" height=\"%d\" alt=\"alerts\"></td></tr>\n"
			"<tr><td>&nbsp;</td></tr>\n"
			"<tr><td>Interface status</td></tr>\n"
			"<tr><td nowrap><img src=\"%s/%s/%d/status.png\" width=\"%d\" height=\"%d\" alt=\"status\"></td></tr>\n",
			root,router[rc].hostname,cur->idx,width[DIM_IO],height[DIM_IO],
			root,router[rc].hostname,cur->idx,width[DIM_ALERTS],height[DIM_ALERTS],
			root,router[rc].hostname,cur->idx,width[DIM_STATUS],height[DIM_STATUS]);

   PIECE(5);

   close (fd);

   return (NULL);
}

/*
 * Generate graphs from the relevant RRD databases and create a web page
 * containing those graphs. Return NULL if successful, the error message
 * otherwise.
 *
 *    router     list of routers
 *    n          number of routers in list
 *    rc         index of current router in list
 *    dbdir      database directory
 *    htdir      web page directory
 *    root       document root (relative to htdir)
 */
const char *html_generate (const router_t *router,int n,int rc,const char *dbdir,const char *htdir,const char *root)
{
   char *filename,*database,*middle,*end[2];
   int fd,width[3],height[3];
   ssize_t result;
   struct stat sb;
   interface_t *iface;
   const char *msg;

   if ((filename = (char *) mem_alloc ((strlen (htdir) + strlen (router[rc].hostname) + 64) * sizeof (char))) == NULL)
	 return (html_set_error ("out of memory: %s",strerror (errno)));

   strcpy (filename,htdir);
   if (*filename != '\0') strcat (filename,"/");
   middle = filename + strlen (filename);
   strcat (middle,"/images");

   mkdirhier (filename);

   strcat (middle,"/");
   end[0] = middle + strlen (middle);

   strcpy (end[0],"seperator.png");
   if (stat (filename,&sb) < 0)
	 {
		if ((fd = creat (filename,0644)) < 0)
		  {
			 fail0:
			 html_set_error ("couldn't create %s: %s",filename,strerror (errno));
			 mem_free (filename);
			 return (html_get_error ());
		  }

		if ((result = write (fd,seperator,NB_OF (seperator))) < 0)
		  {
			 fail1:
			 html_set_error ("write error while writing to %s: %s",filename,strerror (errno));
			 mem_free (filename);
			 return (html_get_error ());
		  }

		if (result != NB_OF (seperator))
		  {
			 html_set_error ("write error while writing to %s: expected %u bytes, wrote %u instead",NB_OF (seperator),result);
			 mem_free (filename);
			 return (html_get_error ());
		  }

		close (fd);
	 }

   strcpy (end[0],"orca.png");
   if (stat (filename,&sb) < 0)
	 {
		if ((fd = creat (filename,0644)) < 0)
		  goto fail0;

		if ((result = write (fd,orca,NB_OF (orca))) < 0)
		  goto fail1;

		if (result != NB_OF (orca))
		  {
			 mem_free (filename);
			 return (html_set_error ("write error while writing to %s: expected %u bytes, wrote %u instead",NB_OF (orca),result));
		  }

		close (fd);
	 }

   if ((database = (char *) mem_alloc ((strlen (dbdir) + strlen (router[rc].hostname) + 64) * sizeof (char))) == NULL)
	 {
		html_set_error ("out of memory: %s",strerror (errno));
		mem_free (filename);
		return (html_get_error ());
	 }

   strcpy (middle,router[rc].hostname);
   strcat (middle,"/");
   middle += strlen (middle);

   strcpy (database,dbdir);
   if (*database != '\0') strcat (database,"/");
   strcat (database,router[rc].hostname);
   strcat (database,"/");
   end[1] = database + strlen (database);

   for (iface = router[rc].iface; iface != NULL; iface = iface->next)
	 if (iface->status[IF_ADMIN_STATUS] == IF_STATUS_UP || !router[rc].ignore_admin_down)
	   {
		  sprintf (middle,"%d/",iface->idx);
		  end[0] = middle + strlen (middle);
		  mkdirhier (filename);

		  sprintf (end[1],"%d.rrd",iface->idx);

		  log_printf (LOG_DEBUG,"generating graphs from RRD database %s\n",database);

		  strcpy (end[0],"io.png");
		  log_printf (LOG_DEBUG,"   %s\n",filename);
		  if ((msg = db_graph_io (&width[DIM_IO],&height[DIM_IO],filename,database)) != NULL)
			{
			   mem_free (filename);
			   mem_free (database);
			   return (msg);
			}

		  strcpy (end[0],"alerts.png");
		  log_printf (LOG_DEBUG,"   %s\n",filename);
		  if ((msg = db_graph_alerts (&width[DIM_ALERTS],&height[DIM_ALERTS],filename,database)) != NULL)
			{
			   mem_free (filename);
			   mem_free (database);
			   return (msg);
			}

		  strcpy (end[0],"status.png");
		  log_printf (LOG_DEBUG,"   %s\n",filename);
		  if ((msg = db_graph_status (&width[DIM_STATUS],&height[DIM_STATUS],filename,database)) != NULL)
			{
			   mem_free (filename);
			   mem_free (database);
			   return (msg);
			}

		  strcpy (end[0],"index.html");
		  log_printf (LOG_DEBUG,"generating html %s\n",filename);
		  if ((msg = gen_html (filename,root,width,height,router,n,rc,iface->idx)) != NULL)
			{
			   mem_free (filename);
			   mem_free (database);
			   return (msg);
			}
	   }

   mem_free (database);
   mem_free (filename);

   return (NULL);
}

