///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#include <cppunit/extensions/HelperMacros.h>
#include <openvdb/Exceptions.h>
#include <openvdb/math/Mat4.h>
#include <openvdb/math/Maps.h>
#include <openvdb/math/Transform.h>
#include <openvdb/util/MapsUtil.h>


class TestPrePostAPI: public CppUnit::TestCase
{
public:
    CPPUNIT_TEST_SUITE(TestPrePostAPI);

    CPPUNIT_TEST(testMat4);
    CPPUNIT_TEST(testLinearTransform);
    CPPUNIT_TEST_SUITE_END();

    void testMat4();
    void testLinearTransform();

    //void testIsType();
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestPrePostAPI);


void
TestPrePostAPI::testMat4()
{
    using namespace openvdb::math;

    double TOL = 1e-7;


    Mat4d m = Mat4d::identity();
    Mat4d minv = Mat4d::identity();

    // create matrix with pre-API
    // Translate Shear Rotate Translate Scale matrix
    m.preScale(Vec3d(1, 2, 3));
    m.preTranslate(Vec3d(2, 3, 4));
    m.preRotate(X_AXIS, 20);
    m.preShear(X_AXIS, Y_AXIS, 2);
    m.preTranslate(Vec3d(2, 2, 2));

    // create inverse using the post-API
    minv.postScale(Vec3d(1.f, 1.f/2.f, 1.f/3.f));
    minv.postTranslate(-Vec3d(2, 3, 4));
    minv.postRotate(X_AXIS,-20);
    minv.postShear(X_AXIS, Y_AXIS, -2);
    minv.postTranslate(-Vec3d(2, 2, 2));

    Mat4d mtest = minv * m;

    // verify that the results is an identity
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[0][0], 1, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[1][1], 1, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[2][2], 1, TOL);

    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[0][1], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[0][2], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[0][3], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[1][0], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[1][2], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[1][3], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[2][0], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[2][1], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[2][3], 0, TOL);

    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[3][0], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[3][1], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[3][2], 0, TOL);

    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[3][3], 1, TOL);
}


void
TestPrePostAPI::testLinearTransform()
{
    using namespace openvdb::math;

    double TOL = 1e-7;

    Transform::Ptr t = Transform::createLinearTransform(1.f);
    Transform::Ptr tinv = Transform::createLinearTransform(1.f);

    // create matrix with pre-API
    // Translate Shear Rotate Translate Scale matrix
    t->preScale(Vec3d(1, 2, 3));
    t->preTranslate(Vec3d(2, 3, 4));
    t->preRotate(20);
    t->preShear(2, X_AXIS, Y_AXIS);
    t->preTranslate(Vec3d(2, 2, 2));

    // create inverse using the post-API
    tinv->postScale(Vec3d(1.f, 1.f/2.f, 1.f/3.f));
    tinv->postTranslate(-Vec3d(2, 3, 4));
    tinv->postRotate(-20);
    tinv->postShear(-2, X_AXIS, Y_AXIS);
    tinv->postTranslate(-Vec3d(2, 2, 2));


    // test this by verifying that equvilent interal matrix
    // represenations are inverses
    Mat4d m = t->baseMap()->getAffineMap()->getMat4();
    Mat4d minv = tinv->baseMap()->getAffineMap()->getMat4();

    Mat4d mtest = minv * m;
    //std::cout <<" the matrix \n"<<mtest<<std::endl;

    // verify that the results is an identity
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[0][0], 1, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[1][1], 1, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[2][2], 1, TOL);

    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[0][1], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[0][2], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[0][3], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[1][0], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[1][2], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[1][3], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[2][0], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[2][1], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[2][3], 0, TOL);

    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[3][0], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[3][1], 0, TOL);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[3][2], 0, TOL);

    CPPUNIT_ASSERT_DOUBLES_EQUAL(mtest[3][3], 1, TOL);
}

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
