//                                               -*- C++ -*-
/**
 *  @file  t_StrongMaximumTest_other.cxx
 *  @brief The test file of StrongMaximumTest class
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: t_StrongMaximumTest_other.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <iostream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "StandardEvent.hxx"
#include "RandomVector.hxx"
#include "Less.hxx"
#include "Greater.hxx"
#include "NumericalMathFunction.hxx"
#include "IdentityMatrix.hxx"
#include "Normal.hxx"
#include "StrongMaximumTest.hxx"

using namespace OT;
using namespace OT::Test;

void printSample(String name, NumericalSample sample)
{
  OStream fullprint(std::cout);

  fullprint << name << "=[" ;
  for (UnsignedLong counter = 0; counter < sample.getSize(); counter++)
    {
      if (counter != 0)  fullprint << ";";
      fullprint << "[" ;
      NumericalPoint point(sample[counter]);
      for (UnsignedLong coordinate = 0; coordinate < point.getDimension(); coordinate++)
        {
          if (coordinate != 0)  fullprint << ",";
          fullprint <<  point[coordinate];
        }
      fullprint << "]" ;
    }
  fullprint << "];" << std::endl;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);

  try {

    /* We create a numerical math function */
    NumericalMathFunction myFunction("TestStrongMaximumTest2");

    UnsignedLong dim(myFunction.getInputDimension());
    /* We create a normal distribution point of dimension 1 */
    NumericalPoint mean(dim, 0.0);
    NumericalPoint sigma(dim, 1.0);
    IdentityMatrix R(dim);
    Normal myDistribution(mean, sigma, R);

    /* We create a 'usual' RandomVector from the Distribution */
    RandomVector vect(myDistribution);

    /* We create a composite random vector */
    RandomVector output(myFunction, vect);

    /* We create a StandardEvent from this RandomVector  : RandomVector > 0.0 */
    double seuil(40);
    StandardEvent myStandardEvent(output, Greater(), seuil);

    /* We create the design point */
    NumericalPoint designPoint(dim, 0.0);
    double C(0.6);
    designPoint[0] = - sqrt(seuil) + C;

    /* We create the "second" design point */
    NumericalPoint pseudoDesignPoint(dim, 0.0);
    pseudoDesignPoint[0] = sqrt(seuil) + C;

    NumericalScalar importanceLevel(0.01);
    NumericalScalar accuracyLevel(2);
    NumericalScalar confidenceLevel(0.999);

    StrongMaximumTest myTest(myStandardEvent, designPoint, importanceLevel, accuracyLevel, confidenceLevel);
    fullprint << "myTest=" << myTest << std::endl;
    myTest.run();
    fullprint << "Beta = " << designPoint.norm() << std::endl;
    fullprint << "Discretised sphere radius = "  << designPoint.norm() * (1 + myTest.getAccuracyLevel() * myTest.getDeltaEpsilon()) << std::endl;
    fullprint << "PointNumber = " << myTest.getPointNumber() << std::endl;
    fullprint << "DesignPointVicinity Angle (rad)= " << acos(myTest.getDesignPointVicinity()) << std::endl;
    fullprint << "DesignPointVicinity Angle (deg)= " << acos(myTest.getDesignPointVicinity())*180/M_PI << std::endl;
    fullprint << "Near Design Point Verifying Event Points Number = " << (myTest.getNearDesignPointVerifyingEventPoints()).getSize() << std::endl;
    fullprint << "Near Design Point Violating Event Points Number = " << (myTest.getNearDesignPointViolatingEventPoints()).getSize() << std::endl;
    fullprint << "Far Design Point Verifying Event Points Number = " << (myTest.getFarDesignPointVerifyingEventPoints()).getSize() << std::endl;
    fullprint << "Far Design Point Violating Event Points Number = " << (myTest.getFarDesignPointViolatingEventPoints()).getSize() << std::endl;
    fullprint << "//////" << std::endl;
    // parameters of the test
    fullprint << "importanceLevel=" << myTest.getImportanceLevel()<< std::endl;
    fullprint << "accuracyLevel=" << myTest.getAccuracyLevel()<< std::endl;
    fullprint << "confidenceLevel=" << myTest.getConfidenceLevel()<< std::endl;

    // design point coordinates
    fullprint << "standardSpaceDesignPoint=[" << myTest.getStandardSpaceDesignPoint()[0] << "," << myTest.getStandardSpaceDesignPoint()[1] << "]" << std::endl;

    // pseudo design point coordinates
    fullprint << "pseudoStandardSpaceDesignPoint=[" << pseudoDesignPoint[0] << "," << pseudoDesignPoint[1] << "]" << std::endl;

    // cout of the coordinates of the points of the 4 samples
    /* NearDesignPointVerifyingEventPoints */
    printSample("NearDesignPointVerifyingEventPointsSample",myTest.getNearDesignPointVerifyingEventPoints());
    /* NearDesignPointViolatingEventPoints */
    printSample("NearDesignPointViolatingEventPoints",myTest.getNearDesignPointViolatingEventPoints());
    /* FarDesignPointVerifyingEventPoints */
    printSample("FarDesignPointVerifyingEventPoints",myTest.getFarDesignPointVerifyingEventPoints());
    /* FarDesignPointViolatingEventPoints */
    printSample("FarDesignPointViolatingEventPoints",myTest.getFarDesignPointViolatingEventPoints());

    //    fullprint << "myTest (after run())=" << myTest << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
