//                                               -*- C++ -*-
/**
 *  @file  ConditionalRandomVector.cxx
 *  @brief An implementation class for distribution-based random vectors
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2009-09-08 21:20:58 +0200 (mar. 08 sept. 2009) $
 *  Id:      $Id: ConditionalRandomVector.cxx 1320 2009-09-08 19:20:58Z lebrun $
 */
#include "PersistentObjectFactory.hxx"
#include "ConditionalRandomVector.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      typedef Base::Common::InvalidArgumentException InvalidArgumentException;

      CLASSNAMEINIT(ConditionalRandomVector);

      static Base::Common::Factory<ConditionalRandomVector> RegisteredFactory("ConditionalRandomVector");

      /* Default constructor */
      ConditionalRandomVector::ConditionalRandomVector(const Distribution & distribution,
                                                       const RandomVector & randomParameters,
                                                       const String & name)
        : RandomVectorImplementation(name),
          distribution_(distribution),
          randomParameters_(randomParameters)
      {
        // Check if the random parameters random vector has a dimension compatible with
        // the number of parameters of the distribution
        if (randomParameters.getDimension() != distribution.getParametersNumber()) throw InvalidArgumentException(HERE) << "Error: the random parameters dimension must be equal with the number of parameters of the distribution.";
        setDescription(distribution.getDescription());
      }

      /* Virtual constructor */
      ConditionalRandomVector * ConditionalRandomVector::clone() const
      {
        return new ConditionalRandomVector(*this);
      }

      /* String converter */
      String ConditionalRandomVector::__repr__() const {
        OSS oss;
        oss << "class=" << ConditionalRandomVector::GetClassName()
            << " distribution=" << distribution_
            << " random parameters=" << randomParameters_;
        return oss;
      }



      /* Here is the interface that all derived class must implement */


      /* Dimension accessor */
      UnsignedLong ConditionalRandomVector::getDimension() const
      {
        return distribution_.getDimension();
      }

      /* Realization accessor */
      ConditionalRandomVector::NumericalPoint ConditionalRandomVector::getRealization() const
      {
        NumericalPoint parameters;
        return getRealization(parameters);
      }

      /* Realization accessor */
      ConditionalRandomVector::NumericalPoint ConditionalRandomVector::getRealization(NumericalPoint & parameters) const
      {
        parameters = randomParameters_.getRealization();
        distribution_.setParametersCollection(parameters);
        return distribution_.getRealization();
      }

      /* Distribution accessor */
      Distribution ConditionalRandomVector::getDistribution() const
      {
        return distribution_;
      }

      /* Random parameters accessor */
      RandomVector ConditionalRandomVector::getRandomParameters() const
      {
        return randomParameters_;
      }

      /* Method save() stores the object through the StorageManager */
      void ConditionalRandomVector::save(StorageManager::Advocate & adv) const
      {
        RandomVectorImplementation::save(adv);
        adv.saveAttribute( "distribution_", distribution_ );
        adv.saveAttribute( "randomParameters_", randomParameters_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void ConditionalRandomVector::load(StorageManager::Advocate & adv)
      {
        RandomVectorImplementation::load(adv);
        adv.loadAttribute( "distribution_", distribution_ );
        adv.loadAttribute( "randomParameters_", randomParameters_ );
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
