//                                               -*- C++ -*-
/**
 *  @file  Rayleigh.hxx
 *  @brief The Rayleigh distribution
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-09-09 23:26:29 +0200 (mar, 09 sep 2008) $
 *  Id:      $Id: Rayleigh.hxx 924 2008-09-09 21:26:29Z lebrun $
 */
#ifndef OPENTURNS_RAYLEIGH_HXX
#define OPENTURNS_RAYLEIGH_HXX

#include "NonEllipticalDistribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Rayleigh
       *
       * The Rayleigh distribution.
       */
      class Rayleigh
        : public Model::NonEllipticalDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution; // required by SWIG
        typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
        typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
        typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
        typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
        typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;
        typedef NonEllipticalDistribution::StorageManager            StorageManager;

        /** Default constructor */
        Rayleigh();

        /** Parameters constructor */
        Rayleigh(const NumericalScalar sigma,
                 const NumericalScalar gamma = 0.0)
          /* throw (InvalidArgumentException) */;


        /** Comparison operator */
        Bool operator ==(const Rayleigh & other) const;

        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual Rayleigh * clone() const;

        /** Get one realization of the Rayleigh distribution */
        NumericalPoint getRealization() const;

        /** Get the DDF of the distribution, i.e. the gradient of its PDF w.r.t. point */
        using NonEllipticalDistribution::computeDDF;
        NumericalPoint computeDDF(const NumericalPoint & point) const;

        /** Get the PDF of the distribution, i.e. P(point < X < point+dx) = PDF(point)dx + o(dx) */
        using NonEllipticalDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution, i.e. P(X <= point) = CDF(point). If tail=true, compute P(X >= point) */
        using NonEllipticalDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
        NumericalComplex computeCharacteristicFunction(const NumericalScalar x,
                                                       const Bool logScale = false) const;

        /** Get the gradient of the PDF w.r.t the parameters of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the gradient of the CDF w.r.t the parameters of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const /* throw(NotDefinedException) */;

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const /* throw(NotDefinedException) */;

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const /* throw(NotDefinedException) */;

        /** Get the raw moments of the distribution */
        NumericalPoint getStandardMoment(const UnsignedLong n) const;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;
        using NonEllipticalDistribution::setParametersCollection;
        void setParametersCollection(const NumericalPointCollection & parametersCollection);

        /* Interface specific to Rayleigh */

        /** Sigma accessor */
        void setSigma(const NumericalScalar sigma)
          /* throw(InvalidArgumentException) */;
        NumericalScalar getSigma() const;

        /** Gamma accessor */
        void setGamma(const NumericalScalar gamma);
        NumericalScalar getGamma() const;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);

      protected:

      private:

        /** Compute the mean of the distribution */
        void computeMean() const;

        /** Compute the covariance of the distribution */
        void computeCovariance() const;

        /** Get the quantile of the distribution, i.e the value Xp such that P(X <= Xp) = prob */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const Bool tail = false,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** The dispersion parameter */
        NumericalScalar sigma_;

        /** The position parameter */
        NumericalScalar gamma_;


      }; /* class Rayleigh */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_RAYLEIGH_HXX */
