//                                               -*- C++ -*-
/**
 *  @file  SimulationResult.hxx
 *  @brief SimulationResult stores the simulation result
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-23 13:46:12 +0200 (Fri, 23 May 2008) $
 *  Id:      $Id: SimulationResult.hxx 821 2008-05-23 11:46:12Z schueller $
 */
#ifndef OPENTURNS_SIMULATIONRESULT_HXX
#define OPENTURNS_SIMULATIONRESULT_HXX

#include "SimulationResultImplementation.hxx"
#include "Pointer.hxx"
#include "TypedInterfaceObject.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      /**
       * @class SimulationResult
       * SimulationResult stores the simulation result
       */
      class SimulationResult
        : public Base::Common::TypedInterfaceObject<SimulationResultImplementation>
      {

        CLASSNAME;
      public:

        typedef Pointer<SimulationResultImplementation>                       Implementation;
        typedef SimulationResultImplementation::InvalidArgumentException      InvalidArgumentException;
        typedef SimulationResultImplementation::NumericalPoint                NumericalPoint;
        typedef SimulationResultImplementation::NumericalPointWithDescription NumericalPointWithDescription;
        typedef SimulationResultImplementation::Graph                         Graph;
        typedef SimulationResultImplementation::Event                         Event;

        /** Default constructor */
        explicit SimulationResult();

        /** Standard constructor */
        explicit SimulationResult(const Event & event,
                                  const NumericalScalar probabilityEstimate,
                                  const NumericalScalar varianceEstimate,
                                  const UnsignedLong outerSampling,
                                  const UnsignedLong blockSize,
                                  const NumericalPoint & meanPointInEventDomain) /* throw(InvalidArgumentException) */;

        /** Constructor with parameters */
        SimulationResult(const SimulationResultImplementation & implementation);

        /** Constructor with parameters */
        SimulationResult(Implementation & p_implementation);

        /** Event accessor */
        Event getEvent() const;

        /** Probability estimate accessor */
        NumericalScalar getProbabilityEstimate() const;
        void setProbabilityEstimate(const NumericalScalar probabilityEstimate);

        /** Variance estimate accessor */
        NumericalScalar getVarianceEstimate() const;
        void setVarianceEstimate(const NumericalScalar varianceEstimate);

        /** Mean point conditioned to the event realization accessor */
        NumericalPoint getMeanPointInEventDomain() const;
        void setMeanPointInEventDomain(const NumericalPoint & meanPointInEventDomain);

        /** Importance factors accessor */
        NumericalPointWithDescription getImportanceFactors() const;

        /** Coefficient of variation estimate accessor */
        NumericalScalar getCoefficientOfVariation() const;

        /** Standard deviation estimate accessor */
        NumericalScalar getStandardDeviation() const;

        /** Sample size accessor */
        UnsignedLong getOuterSampling() const;
        void setOuterSampling(const UnsignedLong outerSampling);

        /** Block size accessor */
        UnsignedLong getBlockSize() const;
        void setBlockSize(const UnsignedLong blockSize);

        /** Confidence length */
        NumericalScalar getConfidenceLength(const NumericalScalar level = SimulationResultImplementation::DefaultConfidenceLevel) const
          /* throw(SimulationResultImplementation::InvalidArgumentException) */;

        /** Draw the importance factors */
        Graph drawImportanceFactors() const;

        /** String converter */
        String __repr__() const;

      private:

      }; // class Result

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_SIMULATIONRESULT_HXX */
