//                                               -*- C++ -*-
/**
 *  @file  HermiteFactory.cxx
 *  @brief Hermite polynomial factory
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "HermiteFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "Normal.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(HermiteFactory);

      static Base::Common::Factory<HermiteFactory> RegisteredFactory("HermiteFactory");

      typedef Distribution::Normal                   Normal;
      typedef Base::Common::InvalidArgumentException InvalidArgumentException;

      /* Default constructor */
      HermiteFactory::HermiteFactory()
        : OrthogonalUniVariatePolynomialFactory(Normal())
      {
        initializeCache();
      }


      /* Virtual constructor */
      HermiteFactory * HermiteFactory::clone() const
      {
        return new HermiteFactory(*this);
      }


      /* Calculate the coefficients of recurrence a0n, a1n, a2n such that
         Pn+1(x) = (a0n * x + a1n) * Pn(x) + a2n * Pn-1(x) */
      HermiteFactory::Coefficients HermiteFactory::getRecurrenceCoefficients(const UnsignedLong n) const
      {
        Coefficients recurrenceCoefficients(3, 0.0);
        if (n == 0)
          {
            recurrenceCoefficients[0] = 1.0;
            recurrenceCoefficients[1] = 0.0;
            // Conventional value of 0.0 for recurrenceCoefficients[2]
            return recurrenceCoefficients;
          }
        recurrenceCoefficients[0] = 1.0 / sqrt(n + 1);
        recurrenceCoefficients[1] = 0.0;
        recurrenceCoefficients[2] = -sqrt(1.0 - 1.0 / (n + 1));
        return recurrenceCoefficients;
      }


      /* String converter */
      String HermiteFactory::__repr__() const
      {
        return OSS() << "class=" << getClassName()
                     << " measure=" << measure_;
      }


      /* Method save() stores the object through the StorageManager */
      void HermiteFactory::save(StorageManager::Advocate & adv) const
      {
        OrthogonalUniVariatePolynomialFactory::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void HermiteFactory::load(StorageManager::Advocate & adv)
      {
        OrthogonalUniVariatePolynomialFactory::load(adv);
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
