//                                               -*- C++ -*-
/**
 *  @file  NatafIndependentCopulaHessian.cxx
 *  @brief Class for the Nataf transformation evaluation for independent
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: NatafIndependentCopulaHessian.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include <cmath>
#include "NatafIndependentCopulaHessian.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class NatafIndependentCopulaHessian
       *
       * This class offers an interface for the Nataf function for independent copula
       */

      typedef Distribution::DistFunc DistFunc;

      CLASSNAMEINIT(NatafIndependentCopulaHessian);

      static Base::Common::Factory<NatafIndependentCopulaHessian> RegisteredFactory("NatafIndependentCopulaHessian");

      /* Default constructor */
      NatafIndependentCopulaHessian::NatafIndependentCopulaHessian():
        NumericalMathHessianImplementation(),
        dimension_()
      {
        // Nothing to do
      }

      /* Parameter constructor */
      NatafIndependentCopulaHessian::NatafIndependentCopulaHessian(const UnsignedLong dimension):
        NumericalMathHessianImplementation(),
        dimension_(dimension)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      NatafIndependentCopulaHessian * NatafIndependentCopulaHessian::clone() const
      {
        return new NatafIndependentCopulaHessian(*this);
      }

      /* String converter */
      String NatafIndependentCopulaHessian::__repr__() const
      {
        OSS oss;
        oss << "class=" << NatafIndependentCopulaHessian::GetClassName()
            << " dimension=" << dimension_;
        return oss;
      }

      /*
       * Evaluation
       * The Nataf transform T reads:
       * Ti(xi) = Q(xi), where Q = Phi^{-1} and Phi is the CDF of the standard normal distribution
       * We have:
       * Hijk = d2Ti/dxjdxk = Q''(xi) if i = j = k
       *                    = 0 else
       */
      NatafIndependentCopulaHessian::SymmetricTensor NatafIndependentCopulaHessian::hessian(const NumericalPoint & inP) const
      /* throw(InvalidArgumentException, InternalException) */
      {
        SymmetricTensor result(dimension_, dimension_);
        for (UnsignedLong i = 0; i < dimension_; ++i)
          {
            NumericalScalar x(inP[i]);
            if ((x < 0.0) || (x > 1.0)) throw InvalidArgumentException(HERE) << "Error: cannot evaluate the NatafIndependentCopulaHessian if all the components are not in [0, 1], here in=" << inP;
            // q = Normal(0,1).computeQuantile(x)
            NumericalScalar q(DistFunc::qNormal(x));
            // factor = 1/Normal(0,1).computePDF(q)
            // 6.283185307179586476925286 = 2Pi
            // quantileSecondDerivative = -Normal(0,1).computeDDF(q) / (Normal(0,1).computePDF(q))^3 = q / (Normal(0,1).computePDF(q))^2
            result(i, i, i) = 6.283185307179586476925286 * q * exp(q * q);
          }
        return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong NatafIndependentCopulaHessian::getInputDimension() const
      /* throw(InternalException) */
      {
        return dimension_;
      }

      /* Accessor for output point dimension */
      UnsignedLong NatafIndependentCopulaHessian::getOutputDimension() const
      /* throw(InternalException) */
      {
        return dimension_;
      }

      /* Method save() stores the object through the StorageManager */
      void NatafIndependentCopulaHessian::save(StorageManager::Advocate & adv) const
      {
        NumericalMathHessianImplementation::save(adv);
        adv.saveAttribute( "dimension_", dimension_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void NatafIndependentCopulaHessian::load(StorageManager::Advocate & adv)
      {
        NumericalMathHessianImplementation::load(adv);
        adv.loadAttribute( "dimension_", dimension_ );
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
