//                                               -*- C++ -*-
/**
 *  @file  TensorImplementation.cxx
 *  @brief TensorImplementation implements the Tensor classes
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: TensorImplementation.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include "TensorImplementation.hxx"
#include "PersistentObjectFactory.hxx"
#include "Os.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      CLASSNAMEINIT(TensorImplementation);

      static Common::Factory<TensorImplementation> RegisteredFactory("TensorImplementation");

      typedef Common::Os Os;

      /* Default constructor */
      TensorImplementation::TensorImplementation()
        : PersistentCollection<NumericalScalar>(),
          nbRows_(0),
          nbColumns_(0),
          nbSheets_(0)
      {
        // Nothing to do
      }

      /* Constructor with size (rowDim, colDim and sheetDim) */
      /* The TensorImplementation is made up of a collection of rowDim*colDim*sheetDim elements */
      /* The TensorImplementation is viewed as a set of column vectors read one after another, one sheet after another */
      TensorImplementation::TensorImplementation(const UnsignedLong rowDim,
                                                 const UnsignedLong colDim,
                                                 const UnsignedLong sheetDim)
        : PersistentCollection<NumericalScalar>(rowDim * colDim * sheetDim, 0.0),
          nbRows_(rowDim),
          nbColumns_(colDim),
          nbSheets_(sheetDim)
      {
        // Nothing to do
      }

      /* Constructor from external collection */
      TensorImplementation::TensorImplementation(const UnsignedLong rowDim,
                                                 const UnsignedLong colDim,
                                                 const UnsignedLong sheetDim,
                                                 const Collection<NumericalScalar> & elementsValues)
        : PersistentCollection<NumericalScalar>(rowDim * colDim * sheetDim, 0.0),
          nbRows_(rowDim),
          nbColumns_(colDim),
          nbSheets_(sheetDim)
      {
        const UnsignedLong tensorSize = std::min(rowDim * colDim * sheetDim, elementsValues.getSize());
        for(UnsignedLong i = 0; i < tensorSize; ++i)
          {
            (*this)[i] = elementsValues[i];
          }
      }


      /* Virtual constructor */
      TensorImplementation * TensorImplementation::clone() const
      {
        return new TensorImplementation(*this);
      }


      /* String converter */
      String TensorImplementation::__repr__() const
      {
        return OSS() << "class=" << getClassName()
                     << " name=" << getName()
                     << " rows=" << getNbRows()
                     << " columns=" << getNbColumns()
                     << " sheets=" << getNbSheets()
                     << " values=" << PersistentCollection<NumericalScalar>::__repr__();
      }

      String TensorImplementation::__str__(const String & offset) const
      {
        OSS oss(false);
        for (UnsignedLong k = 0; k < getNbSheets(); ++k)
          {
            oss << (k == 0 ? "" : Os::GetEndOfLine()) << offset << "sheet " << k << ":" << Os::GetEndOfLine();
            oss << offset << getSheet(k).__str__(offset) << Os::GetEndOfLine();
          }
        return oss;
      }

      /* Get the dimensions of the TensorImplementation : number of rows */
      const UnsignedLong TensorImplementation::getNbRows() const
      {
        return nbRows_;
      }

      /* Get the dimensions of the TensorImplementation : number of columns */
      const UnsignedLong TensorImplementation::getNbColumns() const
      {
        return nbColumns_;
      }

      /* Get the dimensions of the TensorImplementation : number of sheets */
      const UnsignedLong TensorImplementation::getNbSheets() const
      {
        return nbSheets_;
      }

      /* Operator () gives access to the elements of the TensorImplementation (to modify these elements) */
      /* The element of the TensorImplementation is designated by its row number i, its column number j and its sheet number k */
      /* the first element of the TensorImplementation is t(0,0,0) */
      NumericalScalar & TensorImplementation::operator() (const UnsignedLong i,
                                                          const UnsignedLong j,
                                                          const UnsignedLong k) /* throw(InvalidDimensionException) */
      {
        if ((i >= nbRows_) || (j >= nbColumns_) || (k >= nbSheets_)) throw InvalidDimensionException(HERE);

        return (*this)[this->convertPosition(i, j, k)];
      }

      /* Operator () gives access to the elements of the TensorImplementation (read only) */
      /* The element of the TensorImplementation is designated by its row number i, its column number j and its sheet number k */
      const NumericalScalar & TensorImplementation::operator() (const UnsignedLong i,
                                                                const UnsignedLong j,
                                                                const UnsignedLong k)  const /* throw(InvalidDimensionException) */
      {
        if ((i >= nbRows_) || (j >= nbColumns_) || (k >= nbSheets_)) throw InvalidDimensionException(HERE);

        return (*this)[this->convertPosition(i,j,k)];
      }

      /* getSheet returns the sheet specified by its sheet number k */
      Matrix TensorImplementation::getSheet(const UnsignedLong k) const /* throw(InvalidDimensionException) */
      {
        if (k >= nbSheets_) throw InvalidDimensionException(HERE);

        Matrix sheet(nbRows_, nbColumns_);
        for (UnsignedLong i = 0; i < nbRows_; ++i)
          for (UnsignedLong j = 0; j < nbColumns_; ++j)
            sheet(i, j) = (*this)(i, j, k);

        return sheet;
      }

      /* setSheet sets matrix m as the sheet specified by its sheet number k  */
      void TensorImplementation::setSheet(const UnsignedLong k,
                                          const Matrix & m) /* throw(InvalidDimensionException) */
      {
        if (k >= nbSheets_) throw InvalidDimensionException(HERE);

        for (UnsignedLong i = 0; i < nbRows_; ++i)
          for (UnsignedLong j = 0; j < nbColumns_; ++j)
            (*this)(i, j, k) = m(i, j);
      }

      /* getSheetSym returns the symmetric sheet specified by its sheet number k */
      SymmetricMatrix TensorImplementation::getSheetSym(const UnsignedLong k) const /* throw(InvalidDimensionException) */
      {
        if (k >= nbSheets_) throw InvalidDimensionException(HERE);
        if (nbRows_ != nbColumns_) throw InvalidDimensionException(HERE);
        SymmetricMatrix sheet(nbRows_);
        for (UnsignedLong i = 0; i < nbRows_; ++i)
          for (UnsignedLong j = i; j < nbColumns_; ++j)
            sheet(i, j) = (*this)(i, j, k);

        return sheet;
      }

      /* setSheetSym sets symmetric matrix m as the sheet specified by its sheet number k  */
      void TensorImplementation::setSheetSym(const UnsignedLong k,
                                             const SymmetricMatrix & m) /* throw(InvalidDimensionException, InvalidArgumentException) */
      {
        if (k >= nbSheets_) throw InvalidDimensionException(HERE);

        if (m.getDimension() != nbRows_) throw InvalidArgumentException(HERE);
        if (nbRows_ != nbColumns_) throw InvalidDimensionException(HERE);

        for (UnsignedLong i = 0; i < nbRows_; ++i)
          for (UnsignedLong j = i; j < nbColumns_; ++j)
            (*this)(i, j, k) = m(i, j);
      }


      /* Empty returns true if there is no element in the TensorImplementation */
      const Bool TensorImplementation::isEmpty() const
      {
        return ((nbRows_ == 0)  || (nbColumns_ == 0) || (nbSheets_ == 0) || (PersistentCollection<NumericalScalar>::isEmpty())) ;
      }


      /* Symmetrize MatrixImplementation in case it is a symmetric matrix (stored as a triangular matrix) */
      void TensorImplementation::symmetrize() const
      {
        TensorImplementation & refThis = * const_cast<TensorImplementation *>(this);
        for (UnsignedLong k = 0; k < nbSheets_; ++k)
          for (UnsignedLong i = 0; i < nbRows_; ++i)
            for (UnsignedLong j = 0; j < i; ++j)
              refThis[this->convertPosition(i, j, k)] = (*this)[this->convertPosition(j, i, k)];
      }

      /* Comparison operator */
      Bool TensorImplementation::operator == (const TensorImplementation & rhs) const
      {
        const TensorImplementation &lhs(*this);
        Bool equality = true;

        if (&lhs != &rhs) { // Not the same object
          const PersistentCollection<NumericalScalar> & refLhs = static_cast<const PersistentCollection<NumericalScalar> >(lhs);
          const PersistentCollection<NumericalScalar> & refRhs = static_cast<const PersistentCollection<NumericalScalar> >(rhs);

          equality = ( lhs.nbRows_==rhs.nbRows_ && lhs.nbColumns_==rhs.nbColumns_ && lhs.nbSheets_==rhs.nbSheets_ && refLhs==refRhs);
        }

        return equality;
      }

      /* Method save() stores the object through the StorageManager */
      void TensorImplementation::save(StorageManager::Advocate & adv) const
      {
        PersistentCollection<NumericalScalar>::save(adv);
        adv.saveAttribute("nbRows_",    nbRows_);
        adv.saveAttribute("nbColumns_", nbColumns_);
        adv.saveAttribute("nbSheets_", nbSheets_);
      }

      /* Method load() reloads the object from the StorageManager */
      void TensorImplementation::load(StorageManager::Advocate & adv)
      {
        PersistentCollection<NumericalScalar>::load(adv);

        adv.loadAttribute("nbRows_",    nbRows_);
        adv.loadAttribute("nbColumns_", nbColumns_);
        adv.loadAttribute("nbSheets_", nbSheets_);
      }



    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
