//                                               -*- C++ -*-
/**
 *  @file  LowDiscrepancySequenceImplementation.hxx
 *  @brief This class is a top-level class for low discrepancy sequences
 *  simulation algorithms. It delegates to its children the effective
 *  history strategy.
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy$
 *  @date:   $LastChangedDate$
 *  Id:      $Id$
 */
#ifndef OPENTURNS_LOWDISCREPANCYSEQUENCEIMPLEMENTATION_HXX
#define OPENTURNS_LOWDISCREPANCYSEQUENCEIMPLEMENTATION_HXX

#include <stdint.h>

#include "PersistentObject.hxx"
#include "Interval.hxx"
#include "Indices.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "Collection.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {
      /**
       * @class LowDiscrepancySequenceImplementation
       */

      class LowDiscrepancySequenceImplementation
        : public Base::Common::PersistentObject
      {

        CLASSNAME;

      public:

        typedef Type::NumericalPoint NumericalPoint;
        typedef Type::Interval       Interval;
        typedef Type::Indices        Indices;

        /** Constructor with parameters */
        explicit LowDiscrepancySequenceImplementation(const UnsignedLong dimension = 1);

        /** Virtual constructor */
        virtual LowDiscrepancySequenceImplementation * clone() const;

        /** initialize the sequence */
        virtual void initialize(const UnsignedLong dimension);

        /** Dimension accessor */
        UnsignedLong getDimension() const;

        /** Generate a quasi-random vector of numbers uniformly distributed over [0, 1) */
        virtual NumericalPoint generate();

        /** Generate a sample of pseudo-random vectors of numbers uniformly distributed over [0, 1) */
        virtual NumericalSample generate(const UnsignedLong size);

        /** Compute the star discrepancy of a sample uniformly distributed over [0, 1) */
        static NumericalScalar ComputeStarDiscrepancy(const NumericalSample & sample);

        /** String converter */
        virtual String __repr__() const;

      private:
        /** Compute the local discrepancy of a sample, given a multidimensionnal interval */
        static NumericalScalar ComputeLocalDiscrepancy(const NumericalSample & sample,
                                                       const Interval & interval);

      protected:
        typedef Type::Collection<uint64_t>                           UInt64Collection;

        /** Get the needed prime numbers */
        static UInt64Collection GetPrimeNumbers(const Indices & indices);

        /** Compute the n first prime numbers */
        static UInt64Collection ComputeFirstPrimeNumbers(const UnsignedLong n);

        /** Compute the least prime number greater or equal to n */
        static uint64_t ComputeNextPrimeNumber(const UnsignedLong n);

        /** Dimension parameter */
        UnsignedLong dimension_;

      } ; /* class LowDiscrepancySequenceImplementation */

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_LOWDISCREPANCYSEQUENCEIMPLEMENTATION_HXX */
