//                                               -*- C++ -*-
/**
 *  @file  LinearModel.hxx
 *  @brief LinearModel implements the linear model
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: LinearModel.hxx 1866 2011-04-11 10:32:27Z schueller $
 */
#ifndef OPENTURNS_LINEARMODEL_HXX
#define OPENTURNS_LINEARMODEL_HXX

#include "PersistentObject.hxx"
#include "Description.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "Exception.hxx"
#include "ConfidenceInterval.hxx"
#include "PersistentCollection.hxx"
#include "Collection.hxx"
#include "TestResult.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {

      /**
       * @class LinearModel
       *
       * LinearModel implements the notion of linear model
       */

      class LinearModel :
        public Common::PersistentObject
      {
        CLASSNAME;

      public:

        typedef Common::InvalidArgumentException                     InvalidArgumentException;
        typedef Type::Description                                    Description;
        typedef Type::NumericalPoint                                 NumericalPoint;
        typedef Type::Collection<Stat::ConfidenceInterval>           ConfidenceIntervalCollection;
        typedef Type::PersistentCollection<Stat::ConfidenceInterval> ConfidenceIntervalPersistentCollection;
        typedef Type::Collection<NumericalScalar>                    NumericalScalarCollection;
        typedef Type::PersistentCollection<NumericalScalar>          NumericalScalarPersistentCollection;
        typedef Common::StorageManager                               StorageManager;


        /** Default constructor is private */
        LinearModel();

        /** Standard constructor */
        LinearModel(const NumericalPoint & vectorR,
                    const ConfidenceIntervalCollection & intervalsColl,
                    const NumericalScalarCollection & pValuesOfR) /* throw(InvalidArgumentException) */;

        /** Constructor from NumericalPoint */
        LinearModel(const NumericalPoint & vectorR);

        /** Virtual constructor */
        virtual LinearModel * clone() const;


        /** String converter */
        String __repr__() const;
        String __str__(const String & offset = "") const;

        /** get vector, get intervals*/
        const NumericalPoint & getRegression() const;
        const ConfidenceIntervalCollection & getConfidenceIntervals() const;
        const NumericalScalarCollection &getPValues() const;

        /** getPredicted : build an sample of predicted values */
        NumericalSample getPredicted(const NumericalSample & predictor) const /* throw(InvalidArgumentException) */;

        /** getResidual */
        NumericalSample getResidual(const NumericalSample & predictor,
                                    const NumericalSample & measured) const /* throw(InvalidArgumentException) */;

        /** Method save() stores the object through the StorageManager */
        void save(StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(StorageManager::Advocate & adv);


      protected:

        /** The estimated vector */
        NumericalPoint regression_;

        /** The confidence intervals */
        ConfidenceIntervalPersistentCollection confidenceIntervals_;

        /** The p-Values  */
        NumericalScalarPersistentCollection pValues_;


      private:

        friend class Common::Factory<LinearModel>;

      }; /* class LinearModel */


      /** Comparison operator */
      Bool operator ==(const LinearModel & lhs, const LinearModel & rhs);

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_LINEARMODEL_HXX */
