//                                               -*- C++ -*-
/**
 *  @file  InverseHaltonSequence.cxx
 *  @brief Implementation of the InverseHalton' sequence
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy$
 *  @date:   $LastChangedDate$
 *  Id:      $Id$
 */
#include <cstdlib>

#include "InverseHaltonSequence.hxx"
#include "ResourceMap.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {

      CLASSNAMEINIT(InverseHaltonSequence);

      typedef Common::ResourceMap              ResourceMap;
      typedef Common::InvalidArgumentException InvalidArgumentException;

      const UnsignedLong InverseHaltonSequence::InitialSeed = 1;//strtoul(ResourceMap::GetInstance().get("InverseHaltonSequence-InitialSeed").c_str(), NULL, 0);

      /* Constructor with parameters */
      InverseHaltonSequence::InverseHaltonSequence(const UnsignedLong dimension) :
        LowDiscrepancySequenceImplementation(dimension)
      {
        initialize(dimension);
      }


      /* Virtual constructor */
      InverseHaltonSequence * InverseHaltonSequence::clone() const
      {
        return new InverseHaltonSequence(*this);
      }


      /* Initialize the sequence */
      void InverseHaltonSequence::initialize(const UnsignedLong dimension)
      {
        if (dimension == 0) throw InvalidArgumentException(HERE) << "Dimension must be > 0.";

        dimension_ = dimension;
        base_ = ComputeFirstPrimeNumbers(dimension);
        seed_ = InitialSeed;
      }

      /* Generate a pseudo-random vector of independant numbers uniformly distributed over [0, 1[ */
      InverseHaltonSequence::NumericalPoint InverseHaltonSequence::generate()
      {
        NumericalPoint realization(dimension_);
        // Loop over the components
        for (UnsignedLong i = 0; i < dimension_; ++i)
          {
            NumericalScalar xI(0.0);
            const uint64_t radix(base_[i]);
            const NumericalScalar inverseRadix(1.0 / radix);
            NumericalScalar inverseRadixN(inverseRadix);
            uint64_t currentSeed(seed_);
            while (currentSeed > 0)
              {
                const UnsignedLong digit(currentSeed % radix);
                if (digit != 0) xI += (radix - digit) * inverseRadixN;
                currentSeed /= radix;
                inverseRadixN *= inverseRadix;
              }
            realization[i] = xI;
          }
        ++seed_;
        return realization;
      }


      /* String converter */
      String InverseHaltonSequence::__repr__() const
      {
        OSS oss;
        oss << "class=" << InverseHaltonSequence::GetClassName()
            << " derived from " << LowDiscrepancySequenceImplementation::__repr__()
            << " base=" << base_
            << " seed=" << seed_;
        return oss;
      }

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
