//                                               -*- C++ -*-
/**
 *  @file  SolverImplementation.cxx
 *  @brief
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-04-11 12:32:27 +0200 (Mon, 11 Apr 2011) $
 *  Id:      $Id: SolverImplementation.cxx 1866 2011-04-11 10:32:27Z schueller $
 */
#include <cstdlib>

#include "SolverImplementation.hxx"
#include "Exception.hxx"
#include "NumericalPoint.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Solver
    {

      CLASSNAMEINIT(SolverImplementation);

      typedef Common::NotYetImplementedException NotYetImplementedException;
      typedef Common::ResourceMap                ResourceMap;
      typedef Type::NumericalPoint               NumericalPoint;

      const NumericalScalar SolverImplementation::DefaultAbsoluteError             = ResourceMap::GetAsNumericalScalar( "SolverImplementation-DefaultAbsoluteError" );
      const NumericalScalar SolverImplementation::DefaultRelativeError             = ResourceMap::GetAsNumericalScalar( "SolverImplementation-DefaultRelativeError" );
      const UnsignedLong    SolverImplementation::DefaultMaximumFunctionEvaluation = ResourceMap::GetAsUnsignedLong( "SolverImplementation-DefaultMaximumFunctionEvaluation" );

      /** Default constructor */
      SolverImplementation::SolverImplementation():
        PersistentObject(),
        absoluteError_(DefaultAbsoluteError),
        relativeError_(DefaultRelativeError),
        maximumFunctionEvaluation_(DefaultMaximumFunctionEvaluation),
        usedFunctionEvaluation_(0)
      {
        // Nothing to do
      }

      /** Second parameter constructor */
      SolverImplementation::SolverImplementation(const NumericalScalar absoluteError,
                                                 const NumericalScalar relativeError,
                                                 const UnsignedLong maximumFunctionEvaluation):
        PersistentObject(),
        absoluteError_(absoluteError),
        relativeError_(relativeError),
        maximumFunctionEvaluation_(maximumFunctionEvaluation),
        usedFunctionEvaluation_(0)
      {
        // Nothing to do
      }

      /** Virtual constructor */
      SolverImplementation * SolverImplementation::clone() const
      {
        return new SolverImplementation(*this);
      }

      /** Comparison operator */
      Bool SolverImplementation::operator ==(const SolverImplementation & other) const
      {
        return (absoluteError_ == other.absoluteError_) && (relativeError_ == other.relativeError_) && (maximumFunctionEvaluation_ == other.maximumFunctionEvaluation_) && (usedFunctionEvaluation_ == other.usedFunctionEvaluation_);
      }

      /** String converter */
      String SolverImplementation::__repr__() const
      {
        OSS oss;
        oss << "class=" << SolverImplementation::GetClassName()
            << " absoluteError=" << absoluteError_
            << " relativeError=" << relativeError_
            << " maximumFunctionEvaluation=" << maximumFunctionEvaluation_
            << " usedFunctionEvaluation=" << usedFunctionEvaluation_;
        return oss;
      }

      /** Absolute error accessor */
      void SolverImplementation::setAbsoluteError(const NumericalScalar absoluteError)
      {
        absoluteError_ = absoluteError;
      }

      NumericalScalar SolverImplementation::getAbsoluteError() const
      {
        return absoluteError_;
      }

      /** Relative error accessor */
      void SolverImplementation::setRelativeError(const NumericalScalar relativeError)
      {
        relativeError_ = relativeError;
      }

      NumericalScalar SolverImplementation::getRelativeError() const
      {
        return relativeError_;
      }

      /** Maximum function evaluation accessor */
      void SolverImplementation::setMaximumFunctionEvaluation(const UnsignedLong maximumFunctionEvaluation)
      {
        maximumFunctionEvaluation_ = maximumFunctionEvaluation;
      }

      UnsignedLong SolverImplementation::getMaximumFunctionEvaluation() const
      {
        return maximumFunctionEvaluation_;
      }

      /** Used function evaluation accessor */
      void SolverImplementation::setUsedFunctionEvaluation(const UnsignedLong usedFunctionEvaluation)
      {
        usedFunctionEvaluation_ = usedFunctionEvaluation;
      }

      UnsignedLong SolverImplementation::getUsedFunctionEvaluation() const
      {
        return usedFunctionEvaluation_;
      }

      /** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] */
      NumericalScalar SolverImplementation::solve(const NumericalMathFunction & function,
                                                  const NumericalScalar value,
                                                  const NumericalScalar infPoint,
                                                  const NumericalScalar supPoint)
      /* throw(InternalException, InvalidDimensionException) */
      {
        if ((function.getInputDimension() != 1) || (function.getOutputDimension() != 1))
          {
            throw InvalidDimensionException(HERE) << "Error: solver implementation requires a scalar function, here input dimension=" << function.getInputDimension() << " and output dimension=" << function.getOutputDimension();
          }
        if (maximumFunctionEvaluation_ < 2)
          {
            throw InternalException(HERE) << "Error: solver needs to evaluate the function at least two times, here maximumFunctionEvaluation=" << maximumFunctionEvaluation_;
          }
        /* We take into account the fact that we use 2 function calls when using the other solve method */
        maximumFunctionEvaluation_ -= 2;
        NumericalScalar root(solve(function, value, infPoint, supPoint, function(NumericalPoint(1, infPoint))[0], function(NumericalPoint(1, supPoint))[0]));
        maximumFunctionEvaluation_ += 2;
        usedFunctionEvaluation_ +=2;
        return root;
      }

      /** Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] given function(infPoint) and function(supPoint) */
      NumericalScalar SolverImplementation::solve(const NumericalMathFunction & function,
                                                  const NumericalScalar value,
                                                  const NumericalScalar infPoint,
                                                  const NumericalScalar supPoint,
                                                  const NumericalScalar infValue,
                                                  const NumericalScalar supValue)
      /* throw(InternalException, InvalidDimensionException) */
      {
        throw NotYetImplementedException(HERE);
      }

    } /* namespace Solver */
  } /* namespace Base */
} /* namespace OpenTURNS */
