//                                               -*- C++ -*-
/**
 *  @file  WrapperData.hxx
 *  @brief This class declares the wrapper data that are exchanged with the platform
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: souchaud $
 *  @date:   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 *  Id:      $Id: WrapperData.hxx 1981 2011-07-01 08:34:36Z souchaud $
 */
#ifndef OPENTURNS_WRAPPERDATA_HXX
#define OPENTURNS_WRAPPERDATA_HXX

#include "OTprivate.hxx"
#include "Object.hxx"
#include "Pointer.hxx"           // for shared pointer
#include "WrapperInterface.h"
#include "Collection.hxx"

namespace OpenTURNS
{

  /**
   * @struct WrapperSymbolProvided
   * @brief  Stores information about the presence or no of symbols in the wrapper
   * @ingroup Wrapper
   */
  struct WrapperSymbolProvided
  {
    typedef Bool Value;
    static const Value NO  = WRAPPER_NO;  ///< No symbol is provided
    static const Value YES = WRAPPER_YES; ///< The symbol is provided
  };


  /**
   * @struct WrapperComputedGradient
   * @brief  Stores information about the computation of gradient for a variable
   * @ingroup Wrapper
   * @internal
   * @deprecated
   * This class is deprecated and should not be used anymore.
   */
  struct WrapperComputedGradient
  {
    typedef Bool Value;
    static const Value NO  = WRAPPER_NO;  ///< No need to compute the gradient within the numerical function
    static const Value YES = WRAPPER_YES; ///< We need to compute the gradient within the numerical function
  };


  /**
   * @struct WrapperFunctionDescription
   * @brief  Describes what is the interface of a numerical function
   * @ingroup Wrapper
   */
  struct WrapperFunctionDescription
    : public Base::Common::Object
  {
    String name_;                            ///< The name of the numerical function
    WrapperSymbolProvided::Value provided_;  ///< Is the symbol present in the wrapper ?

    WrapperFunctionDescription();
    virtual String __repr__() const;
    virtual String __str__(const String & offset = "") const;
  };


  /**
   * @struct WrapperDataFileType
   * @brief  Stores the type (input or output) of the file
   * @ingroup Wrapper
   */
  struct WrapperDataFileType
  {
    typedef unsigned long Value;
    static const Value IN  = WRAPPER_IN;  ///< The file is an input file
    static const Value OUT = WRAPPER_OUT; ///< The file is an output file
  };

  /**
   * @struct WrapperDataFile
   * @brief  Holds the name and the path of any file passed as argument to the wrapper
   * @ingroup Wrapper
   */
  struct WrapperDataFile
    : public Base::Common::Object
  {
    String id_;                       ///< The id of the file (any string to distinguish each file from another)
    String name_;                     ///< The name of the file (stdin, stdout, log, etc.)
    FileName path_;                   ///< The path of the file (/tmp/stdin, /var/log/mylog, etc.)
    String subst_;                    ///< The list of variable's ids to be substituted in the file
    WrapperDataFileType::Value type_; ///< The type of the file (input or output)

    WrapperDataFile();
    Bool operator == (const WrapperDataFile & other) const;
    virtual String __repr__() const;
    virtual String __str__(const String & offset = "") const;
  }; /* struct WrapperDataFile */

  /**
   * @struct WrapperFrameworkData
   * @brief  Holds information used by englobing framework (like Salome)
   * @ingroup Wrapper
   * @internal
   * @deprecated
   * This structure will be refactored, so don't rely on this definition.
   */
  struct WrapperFrameworkData
    : public Base::Common::Object
  {
    UnsignedLong studyid_;            ///< The id of the study
    String studycase_;                ///< The entry of the selected case in Salome study
    String componentname_;            ///< The name of the solver component in Salome

    WrapperFrameworkData();
    virtual String __repr__() const;
    virtual String __str__(const String & offset = "") const;
  }; /* struct WrapperFrameworkData */




  /**
   * @struct WrapperDataVariableType
   * @brief  Stores the type (input or output) of the variable
   * @ingroup Wrapper
   */
  struct WrapperDataVariableType {
    typedef unsigned long Value;
    static const Value IN       = WRAPPER_IN;       ///< The variable is an input variable
    static const Value OUT      = WRAPPER_OUT;      ///< The variable is an output variable
    static const Value INTERNAL = WRAPPER_INTERNAL; ///< The variable is internal to the platform (not user defined)
  };

  /**
   * @struct WrapperDataVariableLocation
   * @brief  Stores the line number or the regular expression that locates the variable in the file
   * @ingroup Wrapper
   */
  struct WrapperDataVariableLocation {
    typedef unsigned long Value;
    static const Value LINE   = WRAPPER_LINE;   ///< The location is a line number
    static const Value REGEXP = WRAPPER_REGEXP; ///< The location is a regular expression
  };

  /**
   * @struct WrapperDataVariable
   * @brief  Holds information of any variable passed as argument to the wrapper
   * @ingroup Wrapper
   */
  struct WrapperDataVariable
    : public Base::Common::Object
  {
    String id_;                                   ///< The id of the variable (any string to distinguish each variable from another)
    String comment_;                              ///< A comment to inform on the variable
    String unit_;                                 ///< The unit which the variable is expressed in
    String regexp_;                               ///< The regular expression used to find the variable location in files
    String format_;                               ///< The format in which the variable must be printed in files
    WrapperDataVariableType::Value type_;         ///< The type of the variable (input or output)
    WrapperComputedGradient::Value gradient_;     ///< The gradient of this variable is computed if true (@deprecated)
    WrapperDataVariableLocation::Value fromType_; ///< The type of the from_ member
    String from_;                                 ///< The location where substitution should start
    WrapperDataVariableLocation::Value toType_;   ///< The type of the to_ member
    String to_;                                   ///< The location where substitution should stop

    WrapperDataVariable();
    Bool operator == (const WrapperDataVariable & other) const;
    virtual String __repr__() const;
    virtual String __str__(const String & offset = "") const;
  }; /* struct WrapperDataVariable */




  /**
   * @struct WrapperState
   * @brief  Stores the mode of invocation of the external code
   * @ingroup Wrapper
   */
  struct WrapperState
  {
    typedef unsigned long Value;
    static const Value SHARED   = WRAPPER_SHAREDSTATE;   ///< The intenal state is shared amoung function, gradient and hessian
    static const Value SPECIFIC = WRAPPER_SPECIFICSTATE; ///< Function, gradient and hessian have specific internal state
  };


  /**
   * @struct WrapperMode
   * @brief  Stores the mode of invocation of the external code
   * @ingroup Wrapper
   */
  struct WrapperMode
  {
    typedef unsigned long Value;
    static const Value STATICLINK  = WRAPPER_STATICLINK;  ///< The external code is statically linked with the wrapper
    static const Value DYNAMICLINK = WRAPPER_DYNAMICLINK; ///< The external code is dynamically linked with the wrapper
    static const Value FORK        = WRAPPER_FORK;        ///< the external code is a separate shell command
  };


  /**
   * @struct WrapperDataTransfer
   * @brief  Stores the mode of transmission for the arguments
   * @ingroup Wrapper
   */
  struct WrapperDataTransfer
  {
    typedef unsigned long Value;
    static const Value FILES     = WRAPPER_FILES;     ///< The values are transmitted through files
    static const Value PIPE      = WRAPPER_PIPE;      ///< The values are transmitted through a pipe
    static const Value ARGUMENTS = WRAPPER_ARGUMENTS; ///< The values are transmitted as command line arguments
    static const Value SOCKET    = WRAPPER_SOCKET;    ///< The values are transmitted through a socket
    static const Value CORBA     = WRAPPER_CORBA;     ///< The values are transmitted through CORBA
  };


  /**
   * struct WrapperParameter
   * @brief Holds the configuration of the wrapper
   * @ingroup Wrapper
   */
  struct WrapperParameter
    : public Base::Common::Object
  {
    WrapperState::Value        state_; ///< The sharing mode of internal state
    WrapperMode::Value         mode_;  ///< The mode of invocation of the external code
    WrapperDataTransfer::Value in_;    ///< The transmission mode for input arguments
    WrapperDataTransfer::Value out_;   ///< The transmission mode for output arguments
    String command_;                   ///< The command that should invoque the external code according to mode
    String userPrefix_;                ///< The prefix that helps the user to find its compute dir

    WrapperParameter();
    virtual String __repr__() const;
    virtual String __str__(const String & offset = "") const;
  }; /* struct WrapperParameter */



  /**
   * @class WrapperData
   * @brief Declares the wrapper data that are exchanged with the platform
   * @ingroup Wrapper
   */
  class WrapperData
    : public Base::Common::Object
  {
    CLASSNAME;
  public:

    typedef       struct WrapperExchangedData        ExchangedData;
    typedef       struct WrapperExchangedData *      ExchangedDataPointer;
    typedef const struct WrapperExchangedData * ConstExchangedDataPointer;

    typedef Base::Type::Collection<WrapperDataFile>     FileListType;
    typedef Base::Type::Collection<WrapperDataVariable> VariableListType;

    /** Default constructor */
    WrapperData();

    /** Constructor from C structure */
    WrapperData(const struct WrapperExchangedData * p_exchangedData);

    /* String converter */
    virtual String __repr__() const;
    virtual String __str__(const String & offset = "") const;

    /** Library path accessor
     * @param path The path of the dynamic library of the wrapper
     */
    void setLibraryPath(const FileName & path);
    /** Library path accessor
     * @return The path of the dynamic library of the wrapper
     */
    FileName getLibraryPath() const;

    /** Function description accessor
     * @param funcDescription The complete description of the numerical function inside the wrapper
     */
    void setFunctionDescription(const WrapperFunctionDescription & funcDescription);
    /** Function description accessor
     * @return The complete description of the numerical function inside the wrapper
     */
    WrapperFunctionDescription getFunctionDescription() const;

    /** Gradient description accessor
     * @param gradDescription The complete description of the numerical function gradient inside the wrapper
     */
    void setGradientDescription(const WrapperFunctionDescription & gradDescription);
    /** Gradient description accessor
     * @return The complete description of the numerical function gradient inside the wrapper
     */
    WrapperFunctionDescription getGradientDescription() const;

    /** Hessian description accessor
     * @param hessDescription The complete description of the numerical function hessian inside the wrapper
     */
    void setHessianDescription(const WrapperFunctionDescription & hessDescription);
    /** Hessian description accessor
     * @return The complete description of the numerical function hessian inside the wrapper
     */
    WrapperFunctionDescription getHessianDescription() const;

    /** Accessor to file list
     * @param fileList The collection of file description for the wrapper
     */
    void setFileList(const FileListType & fileList);
    /** Accessor to file list
     * @return The collection of file description for the wrapper
     */
    const FileListType & getFileList() const;

    /** Conversion method for C interface
     * @return A pointer to a newly allocated C structure that contains the same information as the file list
     * @see FreeFileListForCInterface
     */
    struct WrapperFileList * getNewFileListForCInterface() const;

    /** Frees the memory allocated by getNewFileListForCInterface() method
     * @param fileList A pointer to a newly allocated C structure that contains the same information as the file list
     * @see getNewFileListForCInterface
     */
    static void FreeFileListForCInterface(const struct WrapperFileList * fileList);

    /** Accessor to variable list
     * @param variableList The collection of variable description for the wrapper
     */
    void setVariableList(const VariableListType & variableList);
    /** Accessor to variable list
     * @return The collection of variable description for the wrapper
     */
    const VariableListType & getVariableList() const;

    /** Conversion method for C interface
     * @return A pointer to a newly allocated C structure that contains the same information as the variable list
     * @see FreeVariableListForCInterface
     */
    struct WrapperVariableList * getNewVariableListForCInterface() const;

    /** Frees the memory allocated by getNewVariableListForCInterface() method
     * @param fileList A pointer to a newly allocated C structure that contains the same information as the variable list
     * @see getNewVariableListForCInterface
     */
    static void FreeVariableListForCInterface(const struct WrapperVariableList * variableList);

    /** Accessor to wrapper parameters
     * @param parameters The data needed by the wrapper to link to the external code
     */
    void setParameters(const WrapperParameter & parameters);
    /** Accessor to wrapper parameters
     * @return The data needed by the wrapper to link to the external code
     */
    const WrapperParameter & getParameters() const;

    /** Conversion method for C interface
     * @return A pointer to a newly allocated C structure that contains the same information as the wrapper parameters
     * @see FreeParametersForCInterface
     */
    struct WrapperConfiguration * getNewParametersForCInterface() const;

    /** Frees the memory allocated by getNewParametersForCInterface() method
     * @param parameters A pointer to a newly allocated C structure that contains the same information as the wrapper parameters
     * @see getNewParametersForCInterface
     */
    static void FreeParametersForCInterface(const struct WrapperConfiguration * parameters);

    /** Accessor to englobing framework data
     * @param framework The data needed by the wrapper to control the external code located in the englobing framework
     */
    void setFrameworkData(const WrapperFrameworkData & framework);
    /** Accessor to englobing framework data
     * @return The data needed by the wrapper to control the external code located in the englobing framework
     */
    const WrapperFrameworkData & getFrameworkData() const;

    /** Conversion method for C interface
     * @return A pointer to a newly allocated C structure that contains the same information as framework data
     * @see FreeFrameworkForCInterface
     */
    struct FrameworkData * getNewFrameworkForCInterface() const;

    /** Frees the memory allocated by getNewFrameworkForCInterface() method
     * @param framework A pointer to a newly allocated C structure that contains the same information as framework data
     * @see getNewFrameworkForCInterface
     */
    static void FreeFrameworkForCInterface(const struct FrameworkData * framework);

    /** Builds a newly allocated WrapperExchangedData structure
     * @see FreeWrapperExchangedDataForCInterface
     */
    struct WrapperExchangedData * getNewWrapperExchangedDataForCInterface() const;

    /** Frees the memory allocated by getNewWrapperExchangedDataForCInterface() method
     * @param A pointer to a newly allocated WrapperExchangedData structure
     * @see getNewWrapperExchangedDataForCInterface
     */
    static void FreeWrapperExchangedDataForCInterface(const struct WrapperExchangedData * exchangedData);

    /** Check the correctness of the stored data */
    Bool isValid() const;

    /** Check if variables listed in subst_ are actually defined */
    static void CheckSubstitutedVariables(const WrapperDataFile & file, const VariableListType & variableList);

    /** Convert void pointer to exchanged data structure pointer */
    static      ExchangedDataPointer ConvertOpaquePointer(      void * ptr);
    static ConstExchangedDataPointer ConvertOpaquePointer(const void * ptr);

  protected:

  private:

    /** Where the library that hold the function is located */
    FileName libraryPath_;

    /** The description of the function to be bound to */
    WrapperFunctionDescription function_;

    /** The description of the gradient of the function */
    WrapperFunctionDescription gradient_;

    /** The description of the hessian of the function */
    WrapperFunctionDescription hessian_;

    /** A shared pointer on the list of files that are exchanged between the platform and the wrapper */
    Pointer<FileListType> p_fileList_;

    /** A shared pointer on the list of variables that are exchanged between the platform and the wrapper */
    Pointer<VariableListType> p_variableList_;

    /** A shared pointer on the parameters for the wrapper */
    Pointer<WrapperParameter> p_parameters_;

    /** The data related to the englobing framework */
    Pointer<WrapperFrameworkData> p_framework_;

  }; /* class WrapperData */


} /* namespace OpenTURNS */

#endif /* OPENTURNS_WRAPPERDATA_HXX */
