//                                               -*- C++ -*-
/**
 *  @file  CenteredFiniteDifferenceGradient.cxx
 *  @brief Class for the creation of a numerical math gradient implementation
 *         form a numerical math evaluation implementation by using centered
 *         finite difference formula.
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-08-01 13:42:13 +0200 (Mon, 01 Aug 2011) $
 *  Id:      $Id: CenteredFiniteDifferenceGradient.cxx 2060 2011-08-01 11:42:13Z schueller $
 */
#include <cstdlib>

#include "CenteredFiniteDifferenceGradient.hxx"
#include "PersistentObjectFactory.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Diff {

      CLASSNAMEINIT(CenteredFiniteDifferenceGradient);

      typedef Func::NumericalMathEvaluationImplementation NumericalMathEvaluationImplementation;
      typedef Common::ResourceMap                         ResourceMap;

      static Common::Factory<CenteredFiniteDifferenceGradient> RegisteredFactory("CenteredFiniteDifferenceGradient");

      const NumericalScalar CenteredFiniteDifferenceGradient::DefaultEpsilon = ResourceMap::GetAsNumericalScalar( "CenteredFiniteDifferenceGradient-DefaultEpsilon" );

      /* Default constructor */
      CenteredFiniteDifferenceGradient::CenteredFiniteDifferenceGradient() :
        FiniteDifferenceGradient()
      {
        // Nothing to do
      }

      /* Parameter constructor */
      CenteredFiniteDifferenceGradient::CenteredFiniteDifferenceGradient(const NumericalPoint & epsilon,
                                                                         const EvaluationImplementation & p_evaluation)
        /* throw(InvalidDimensionException) */
        : FiniteDifferenceGradient(epsilon, p_evaluation)

      {
      }

      /* Parameter constructor */
      CenteredFiniteDifferenceGradient::CenteredFiniteDifferenceGradient(const NumericalScalar epsilon,
                                                                         const EvaluationImplementation & p_evaluation)
        /* throw(InvalidDimensionException) */
        : FiniteDifferenceGradient(epsilon, p_evaluation)
      {
      }

      /* Parameter constructor */
      CenteredFiniteDifferenceGradient::CenteredFiniteDifferenceGradient(const FiniteDifferenceStep & step,
                                                                         const EvaluationImplementation & p_evaluation)
        : FiniteDifferenceGradient(step, p_evaluation)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      CenteredFiniteDifferenceGradient * CenteredFiniteDifferenceGradient::clone() const
      {
        return new CenteredFiniteDifferenceGradient(*this);
      }


      /* String converter */
      String CenteredFiniteDifferenceGradient::__repr__() const {
        OSS oss;
        oss << "class=" << CenteredFiniteDifferenceGradient::GetClassName()
            << " name=" << getName()
            << " epsilon=" << getEpsilon().__repr__()
            << " evaluation=" << p_evaluation_->__repr__();
        return oss;
      }

      /* String converter */
      String CenteredFiniteDifferenceGradient::__str__(const String & offset) const {
        return OSS(false) << offset << "CenteredFiniteDifferenceGradient epsilon : " << getEpsilon();
      }

      /* Here is the interface that all derived class must implement */

      /* Gradient () */
      CenteredFiniteDifferenceGradient::Matrix CenteredFiniteDifferenceGradient::gradient(const NumericalPoint & inP) const
      /* throw(InvalidArgumentException,InternalException) */
      {
        const UnsignedLong inputDimension(inP.getDimension());
        NumericalPoint step(finiteDifferenceStep_.operator()(inP));
        if (inputDimension != step.getDimension()) throw InvalidArgumentException(HERE) << "Invalid input dimension";
        /* At which points do we have to compute the evaluation for the centered finite difference. We need 2*dim points. */
        Stat::NumericalSample gridPoints(2 * inputDimension, inP);
        for(UnsignedLong i = 0; i < inputDimension; ++i)
          {
            gridPoints[2 * i][i] += step[i];
            gridPoints[2 * i + 1][i] -= step[i];
          } // For i
        /* Evaluate the evaluation */
        Stat::NumericalSample gridValues(p_evaluation_->operator()(gridPoints));
        /* Compute the gradient */
        Matrix result(p_evaluation_->getInputDimension(), p_evaluation_->getOutputDimension());
        for (UnsignedLong i = 0; i < result.getNbRows(); ++i)
          for (UnsignedLong j = 0; j < result.getNbColumns(); ++j)
            /* result(i, j) = (f_j(x + e_i) - f_j(x - e_i)) / (2 * e_i) ~ df_j / dx_i */
            result(i, j) = (gridValues[2 * i][j] - gridValues[2 * i + 1][j]) / (2.0 * step[i]);
        return result;
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
