//                                               -*- C -*-
/**
 * @file  WrapperCommon.h
 * @brief This file provides objects and functions commonly used by wrappers
 *
 * (C) Copyright 2005-2011 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * @author $LastChangedBy: souchaud $
 * @date   $LastChangedDate: 2011-07-01 10:34:36 +0200 (Fri, 01 Jul 2011) $
 */

#ifndef OPENTURNS_WRAPPERCOMMON_H
#define OPENTURNS_WRAPPERCOMMON_H

#include <WrapperInterface.h>

#if defined(__cplusplus)
extern "C" {
#endif /* defined(__cplusplus) */

  typedef char const * wrapper_CHAR;
  typedef const wrapper_CHAR wrapper_CHARTAB[];

  /* The following declarations MUST be coherent with enumeration declarations in WrapperInterface.h */
  extern wrapper_CHARTAB WrapperListElementTypeAsString;
  extern wrapper_CHARTAB WrapperProvidedAsString;
  extern wrapper_CHARTAB WrapperConfigurationStateAsString;
  extern wrapper_CHARTAB WrapperConfigurationModeAsString;
  extern wrapper_CHARTAB WrapperDataTransferModeAsString;
  extern wrapper_CHARTAB WrapperErrorAsString;
  extern wrapper_CHARTAB WrapperLocationAsString;

  /** Return a string describing the error encountered */
  extern const char * getErrorAsString(enum WrapperErrorCode errorCode);

  /** Register an error message that will be later read by the platform */
  extern void setError(void * p_error, const char * format, ...);

  /** Get the registered error message. The returned string need to be freed to avoid memory leaks */
  extern const char * getError(void * p_error);

  /** Print out the user message passed as argument */
  extern void printUserMessage(const char * functionName, const char * format, ...);

  /* Print out the info message passed as argument */
  extern void printInfoMessage(const char * functionName, const char * format, ...);

  /** Print out the message passed as argument */
  extern void printMessage(const char * functionName, const char * format, ...);

  /** Print out a message telling that we enter a function */
  extern void printEntrance(const char * functionName);

  /** Print out a message telling that we leave a function */
  extern void printExit(const char * functionName);

  /** Print out a message telling which state we are working on */
  extern void printState(const char * functionName, void * p_state);

  /** Print out the content of the WrapperExchangedData structure */
  extern void printWrapperExchangedData(const char * functionName, const struct WrapperExchangedData * p_exchangedData);

  /* Print out the content of the WrapperInformation structure */
  extern void printWrapperInformation(const char * functionName, const struct WrapperInformation * p_info);

  /* Print out the content of a point structure */
  extern void printPoint(const char * functionName, const struct point * p_point);

  /* Print out the content of a sample structure */
  extern void printSample(const char * functionName, const struct sample * p_sample);

  /* Print out the content of a matrix structure */
  extern void printMatrix(const char * functionName, const struct matrix * p_matrix);

  /* Print out the content of a tensor structure */
  extern void printTensor(const char * functionName, const struct tensor * p_tensor);






  /* Get the number of variables of some type (in or out) in the data read from the description file */
  extern unsigned long getNumberOfVariables(const struct WrapperExchangedData * p_exchangedData, unsigned long type);

  /* Get the number of files of some type (in or out) in the data read from the description file */
  extern unsigned long getNumberOfFiles(const struct WrapperExchangedData * p_exchangedData, unsigned long type);

  /* Deep copy of a WrapperExchangedData structure (returns non-zero if error occured) */
  extern long copyWrapperExchangedData(struct WrapperExchangedData ** p_p_new_exchangedData, const struct WrapperExchangedData * p_exchangedData, void * p_error);

  /* Free a WrapperExchangedData structure */
  extern void freeWrapperExchangedData(struct WrapperExchangedData * p_exchangedData);

  /* Append a variable to a WrapperExchangedData structure. Return non zero on failure */
  /* NOTE: the variable pointer is kept by the WrapperExchangedData structure, so don't free it */
  extern long appendVariableToWrapperExchangedData(struct WrapperExchangedData * p_exchangedData,
                                                   struct WrapperVariableListElement * p_variable,
                                                   void * p_error);

  /* Get the command line from the exchanged data structure */
  extern const char * getCommand(const struct WrapperExchangedData * p_exchangedData);

  /* Get the user prefix from the exchanged data structure */
  extern const char * getUserPrefix(const struct WrapperExchangedData * p_exchangedData);

  /* Get the study id from the exchanged data structure */
  extern long getStudyId(const struct WrapperExchangedData * p_exchangedData);

  /* Get the study case from the exchanged data structure */
  extern const char * getStudyCase(const struct WrapperExchangedData * p_exchangedData);

  /* Get the component name from the exchanged data structure */
  extern const char * getComponentName(const struct WrapperExchangedData * p_exchangedData);

  /* Get the number of virtual CPUs from the exchanged data structure */
  extern unsigned long getNumberOfCPUs(const struct WrapperExchangedData * p_exchangedData);

  /* Get the timeout when waiting for output files to be read */
  extern unsigned long getOutputFileTimeout(const struct WrapperExchangedData * p_exchangedData);

  /* Get the number of retries allowed for a failed command */
  extern unsigned long getRunCommandRetries(const struct WrapperExchangedData * p_exchangedData);

  /* Get a pointer to a variable structure. NULL if not found */
  extern const struct WrapperVariableListElement * getVariableById(const struct WrapperExchangedData * p_exchangedData, const char * id);

  /* Get a pointer to a file structure. NULL if not found */
  extern const struct WrapperFileListElement * getFileById(const struct WrapperExchangedData * p_exchangedData, const char * id);


  /* Create a temporary directory. This function internaly allocates the storage of the return value.
   * The symmetric function deleteTemporaryDirectory should be used to avoid memory leak.
   */
  extern char * createTemporaryDirectory(const char * tempDirPrefix, const struct WrapperExchangedData * p_exchangedData, void * p_error);

  /* Delete a temporary directory if executionStatus is zero (ie, no error).
   * This function internaly frees the storage allocated by
   * createTemporaryDirectory, so be careful not to do it yourself. The content of the
   * directory is erased.
   */
  extern void deleteTemporaryDirectory(char * tempDir, long executionStatus, void * p_error);

  /* Change working directory to path */
  extern long changeDirectory(const char * path) DEPRECATED;

  /* Return the name of the current directory in a string. Note that this storage is
   * internaly allocated by the function, so users MUST free it (using free()).
   */
  extern char * getCurrentWorkingDirectory(void * p_error);

  /* Create the files listed in the exchanged data structure in the directory.
   * This action may result in variable substitution in the files according to the
   * regular expressions listed in the exchanged data structure. Returns non-zero if
   * some failure occured.
   */
  extern long createInputFiles(const char * directory, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point, void * p_error);

  /* Read the content of the output listed in the exchanged data structure located in
   * the directory. Returns non-zero if some failure occured. For any output variable
   * not found in output file, the corresponding coordinate in p_point is set to NaN.
   */
  extern long readOutputFiles(const char * directory, const struct WrapperExchangedData * p_exchangedData, struct point * p_point, void * p_error);

  /* Make a command line from the template command and the variable list. The returned value is the new
   * command with variables substituted. This buffer is internaly allocated and MUST be deallocated
   * using free.
   */
  extern char * makeCommandFromTemplate(const char * command, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point, void * p_error);

#ifndef WIN32
  /* Create a batch script in the general temporary directory for use by insulateCommand() */
  extern char * makeScript(char * command, const char * directory, const struct WrapperExchangedData * p_exchangedData, void * p_error);
#endif

  /* Insulate a shell command, so that it can be run in a random temporary directory */
  extern char * insulateCommand(char * command, const char * temporaryDir, const struct WrapperExchangedData * p_exchangedData, void * p_error);

  /* Run an insulated command based on the template read from the wrapper description file */
  extern long runInsulatedCommand(const char * temporaryDir, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point, void * p_error);

#if defined(__cplusplus)
} /* end extern "C" */
#endif /* defined(__cplusplus) */

#endif /* OPENTURNS_WRAPPERCOMMON_H */
