//                                               -*- C++ -*-
/**
 *  @file  ResourceMap.cxx
 *  @brief ResourceMap defines top-most resourceMap strategies
 *
 *  (C) Copyright 2005-2011 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-08-01 13:42:13 +0200 (Mon, 01 Aug 2011) $
 *  Id:      $Id: ResourceMap.cxx 2060 2011-08-01 11:42:13Z schueller $
 */
#include <unistd.h>                 // for sysconf
#include <string.h>                 // for strdup
#include "OTthread.hxx"
#include "OTconfig.hxx"
#include "OSS.hxx"
#include "ResourceMap.hxx"
#include "Exception.hxx"
//#include "Path.hxx"
#include "Log.hxx"
#include "XMLToolbox.hxx"

#ifndef WIN32

#ifndef R_EXECUTABLE
#error "R_EXECUTABLE not defined. Check configuration."
#endif

#ifndef XML_SUPPORTED
#error "XML support is mandatory. Check configuration."
#endif

#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      static const char * ConfigurationFileName = "openturns.conf";
      static const char * RootElementName = "openturns-configuration";
      static const char * XMLTag_value = "value";

      static pthread_mutex_t ResourceMap_InstanceMutex_;
      static ResourceMap * ResourceMap_P_instance_ = 0;

      ResourceMap_init::ResourceMap_init()
      {
        static pthread_once_t ResourceMap_InstanceMutex_once = PTHREAD_ONCE_INIT;
        int rc = pthread_once( &ResourceMap_InstanceMutex_once, ResourceMap::Initialize );
        if (rc != 0) { perror("ResourceMap_init::ResourceMap_init once Initialization failed"); exit(1); }
      }



      ResourceMapInstance::ResourceMapInstance( ResourceMap & rm )  throw() : rm_(rm), lock_(ResourceMap_InstanceMutex_) {}
      ResourceMapInstance::ResourceMapInstance( const ResourceMapInstance & other ) : rm_(other.rm_), lock_(other.lock_) {}
      String ResourceMapInstance::__repr__() const { return rm_.__repr__(); }
      ResourceMap & ResourceMapInstance::lock() throw() { return rm_; }
      const ResourceMap & ResourceMapInstance::lock() const throw() { return rm_; }




      /* Since ResourceMap is a singleton, GetInstance gives access to the object */
      ResourceMapInstance ResourceMap::GetInstance()
      {
#ifdef BOGUS_PTHREAD_LIBRARY
        if (!ResourceMap_P_instance_) {
          ResourceMap_P_instance_ = new ResourceMap;
          assert(ResourceMap_P_instance_);
        }
#endif
        return *ResourceMap_P_instance_;
      }

      void ResourceMap::Initialize()
      {
        pthread_mutexattr_t attr;
        pthread_mutexattr_init( &attr );
        //pthread_mutexattr_settype( &attr, PTHREAD_MUTEX_NORMAL );
        //pthread_mutexattr_settype( &attr, PTHREAD_MUTEX_ERRORCHECK );
        pthread_mutexattr_settype( &attr, PTHREAD_MUTEX_RECURSIVE );
        int rc = pthread_mutex_init( &ResourceMap_InstanceMutex_, &attr );
        if (rc != 0) { perror("ResourceMap::Initialize mutex initialization failed"); exit(1); }
#ifdef BOGUS_PTHREAD_LIBRARY
        ResourceMap_P_instance_ = 0;
#else
        ResourceMap_P_instance_ = new ResourceMap;
        assert(ResourceMap_P_instance_);
#endif
      }



      void ResourceMap::Release()
      {
        delete ResourceMap_P_instance_;
        ResourceMap_P_instance_ = 0;
      }


      /* Get a value in the map */
      String ResourceMap::Get(String key)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        return GetInstance().lock().get( key );
      }

      Bool ResourceMap::GetAsBool(String key)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        return GetInstance().lock().getAsBool( key );
      }

      UnsignedLong ResourceMap::GetAsUnsignedLong(String key)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        return GetInstance().lock().getAsUnsignedLong( key );
      }

      NumericalScalar ResourceMap::GetAsNumericalScalar(String key)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        return GetInstance().lock().getAsNumericalScalar( key );
      }

      char * ResourceMap::GetAsNewCharArray(String key)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        return GetInstance().lock().getAsNewCharArray( key );
      }


      /* Set a value in the map */
      void ResourceMap::Set(String key, String value)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        GetInstance().lock().set( key, value );
      }

      void ResourceMap::SetAsBool(String key, Bool value)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        GetInstance().lock().setAsBool( key, value );
      }

      void ResourceMap::SetAsUnsignedLong(String key, UnsignedLong value)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        GetInstance().lock().setAsUnsignedLong( key, value );
      }

      void ResourceMap::SetAsNumericalScalar(String key, NumericalScalar value)
      {
        //MutexLock lock( ResourceMap_InstanceMutex_ );
        GetInstance().lock().setAsNumericalScalar( key, value );
      }



      /* Default constructor */
      ResourceMap::ResourceMap()
        : map_()
      {
        loadDefaultConfiguration();
        loadConfigurationFile();
      }

      /* Method for retrieving information from the resource map */
      String ResourceMap::get(String key) const
      {
        MapType::const_iterator it = map_.find(key);
        if (it == map_.end()) throw InternalException(HERE) << "Key '" << key << "' is missing in ResourceMap. Report Bug";
        return it->second;
      }

      Bool ResourceMap::getAsBool(String key) const
      {
        Bool value;
        String st = get( key );
        std::istringstream iss( st );
        iss >> value;
        return value;
      }

      UnsignedLong ResourceMap::getAsUnsignedLong(String key) const
      {
        UnsignedLong value;
        String st = get( key );
        std::istringstream iss( st );
        iss >> value;
        return value;
      }

      NumericalScalar ResourceMap::getAsNumericalScalar(String key) const
      {
        NumericalScalar value;
        String st = get( key );
        std::istringstream iss( st );
        iss >> value;
        return value;
      }

      char * ResourceMap::getAsNewCharArray(String key) const
      {
        String st = get( key );
        return strdup( st.c_str() );
      }

      /* Method for setting information into the resource map */
      void ResourceMap::set(String key, String value)
      {
        map_[ key ] = value;
      }

      void ResourceMap::setAsBool(String key, Bool value)
      {
        set( key, OSS() << value );
      }

      void ResourceMap::setAsUnsignedLong(String key, UnsignedLong value)
      {
        set( key, OSS() << value );
      }

      void ResourceMap::setAsNumericalScalar(String key, NumericalScalar value)
      {
        set( key, OSS() << value );
      }


#if defined HAVE_LIBXML2

      /* Update the ResourceMap with information from the configuration file */
      void ResourceMap::readConfigurationFile(const FileName & configurationFile)
      {
        LOGINFO(OSS() << "Using configuration file " << configurationFile);

        // Open the configuration file and load it in a DOM document for later analysis
        XMLDoc document( configurationFile );

        // Check it is an OpenTURNS' one
        xmlNodePtr rootElt = xmlDocGetRootElement( document );
        if (rootElt == NULL) throw ConfigurationFileParsingException(HERE) << "Wrapper file has no root element" << configurationFile;
        if (xmlStrcmp( rootElt->name, REINTERPRET_CAST(const xmlChar *, RootElementName) ))
          throw ConfigurationFileParsingException(HERE) << "Wrapper file " << configurationFile
                                                        << " has an invalid root element (" << rootElt->name << ")"
                                                        << " at line " << xmlGetLineNo(rootElt);


        // Walk through the DOM document to read the data
        if (rootElt && rootElt->children) {
          for (xmlNodePtr current = rootElt->children; current; current = current->next) {
            if (XML::IsElement(current)) {
              String key = XML::ToString(current->name);
              String value = XML::GetAttributeByName( current, XMLTag_value );
              //if (value.empty()) value = XMLAttr_undefined;
              map_[ key ] = value;
            }
          }
        }

      }

#else
      /* Update the ResourceMap with information from the configuration file */
      void ResourceMap::readConfigurationFile(const FileName & configurationFile) {}

#endif

      /* Find the configuration file in specific path (see Path class for algorithm) */
      FileName ResourceMap::findConfigurationFile() const
      {
        Path::DirectoryList dirList = Path::GetConfigDirectoryList();
        return Path::FindFileByNameInDirectoryList(ConfigurationFileName, dirList);
      }

      /* Load the configuration file */
      void ResourceMap::loadConfigurationFile()
      {
        try {
          readConfigurationFile(findConfigurationFile());
        }
        catch (FileNotFoundException & ex) {
          LOGWARN(OSS() << "The configuration file has not been found, using default parameters.");
        }
      }



      /* Load the configuration defined at installation time */
      void ResourceMap::loadDefaultConfiguration()
      {
#ifdef R_EXECUTABLE
        set( "R-executable-command", R_EXECUTABLE );
#endif
        set( "csv-file-separator", ";" );
#ifndef WIN32
        set( "temporary-directory", "/tmp" );
        setAsUnsignedLong( "parallel-threads", sysconf(_SC_NPROCESSORS_CONF) );
#else
        set( "temporary-directory", "TEMP" );
        set( "parallel-threads", getenv("NUMBER_OF_PROCESSORS") );
#endif
        set( "user-prefix", "openturnsComputation" );
        set( "real-regexp-shortcut", "\\R" );
        set( "integer-regexp-shortcut", "\\I" );
        set( "separator-regexp-shortcut", "\\S" );
        set( "real-regexp", "([+-]?[ \t]*([0-9]+(\\.[0-9]*)?|\\.[0-9]+)([Ee][+-]?[0-9]+)?)" );
        set( "integer-regexp", "([+-]?[ \t]*[0-9]+)" );
        set( "separator-regexp", "([ \t])" );
        setAsUnsignedLong( "regexp-shortcut-width", 1 );
        setAsUnsignedLong( "cache-max-size", 1024 );
        setAsUnsignedLong( "output-files-timeout", 2 );
        setAsUnsignedLong( "run-command-retries", 3 );
        setAsUnsignedLong( "slow-filesystem-wait-time", 5000 );
        setAsUnsignedLong( "slow-filesystem-error-recovery", 500000 );
        setAsUnsignedLong( "parallel-execution-grainsize", 1 );
        setAsUnsignedLong( "computation-progression-update-interval", 5 );
        setAsBool( "computation-progression-visible", true );

        // OT parameters
        setAsUnsignedLong( "WrapperFile-version", 1 );

        // OT::Base::Type::Collection parameters
        setAsUnsignedLong( "Collection-size-visible-in-str-from", 10 );

        // OT::Base::Algo::KFold parameters
        setAsUnsignedLong( "KFold-DefaultK", 10 );
                
        // OT::Base::Diff::ProportionalStep parameters //
        setAsNumericalScalar( "BlendedStep-DefaultEta", 0.0 );

        // OT::Base::Diff::CenteredFiniteDifferenceGradient parameters //
        setAsNumericalScalar( "CenteredFiniteDifferenceGradient-DefaultEpsilon", 1.0e-5 );

        // OT::Base::Diff::CenteredFiniteDifferenceHessian parameters //
        setAsNumericalScalar( "CenteredFiniteDifferenceHessian-DefaultEpsilon", 1.0e-4 );

        // OT::Base::Diff::NonCenteredFiniteDifferenceGradient parameters //
        setAsNumericalScalar( "NonCenteredFiniteDifferenceGradient-DefaultEpsilon", 1.0e-7 );

        // OT::Base::Func::SpecFunc parameters //
        setAsNumericalScalar( "SpecFunc-Precision", 1.0e-12 );

        // OT::Base::Graph::Pie parameters //
        setAsNumericalScalar( "Pie-HorizontalMargin", 0.3 );
        setAsNumericalScalar( "Pie-VerticalMargin", 0.1 );
        setAsNumericalScalar( "Pie-LabelThreshold", 0.02 );

        // OT::Base::Graph::DrawableImplementation parameters //
        set( "DrawableImplementation-NoSpecifiedLabel", "" );
        set( "DrawableImplementation-DefaultColor", "blue" );
        set( "DrawableImplementation-DefaultSurfaceColor", "white" );
        set( "DrawableImplementation-DefaultFillStyle", "solid" );
        set( "DrawableImplementation-DefaultPointStyle", "plus" );
        set( "DrawableImplementation-DefaultLineStyle", "solid" );
        set( "DrawableImplementation-DefaultPattern", "s" );
        set( "DrawableImplementation-AlternativePattern", "S" );
        setAsUnsignedLong( "DrawableImplementation-DefaultLineWidth", 1 );
        setAsUnsignedLong( "DrawableImplementation-DataThreshold", 2000 );

        // OT::Base::Graph::GraphImplementation parameters //
        set( "GraphImplementation-NoSpecifiedLabel", "" );
        setAsUnsignedLong( "GraphImplementation-DefaultWidth", 640 );
        setAsUnsignedLong( "GraphImplementation-DefaultHeight", 480 );
        setAsNumericalScalar( "GraphImplementation-DefaultLegendFontSize", 1.0 );

        // OT::Base::Graph::Contour parameters //
        setAsUnsignedLong( "Contour-DefaultLevelsNumber", 10 );

        // OT::Base::Optim::BoundConstrainedAlgorithmImplementation parameters //
        setAsUnsignedLong( "BoundConstrainedAlgorithmImplementation-DefaultMaximumEvaluationsNumber", 100 );
        setAsNumericalScalar( "BoundConstrainedAlgorithmImplementation-DefaultMaximumAbsoluteError", 1.0e-5 );
        setAsNumericalScalar( "BoundConstrainedAlgorithmImplementation-DefaultMaximumRelativeError", 1.0e-5 );
        setAsNumericalScalar( "BoundConstrainedAlgorithmImplementation-DefaultMaximumObjectiveError", 1.0e-5 );
        setAsNumericalScalar( "BoundConstrainedAlgorithmImplementation-DefaultMaximumConstraintError", 1.0e-5 );

        // OT::Base::Optim::SQP::SQP parameters //
        setAsNumericalScalar( "SQP-DefaultTau", 0.5 );
        setAsNumericalScalar( "SQP-DefaultOmega", 1.0e-4 );
        setAsNumericalScalar( "SQP-DefaultSmooth", 1.2 );

        // OT::Base::Optim::TNC::TNC parameters //
        setAsUnsignedLong( "TNC-DefaultMaxCGit", 50 );
        setAsNumericalScalar( "TNC-DefaultEta", 0.25 );
        setAsNumericalScalar( "TNC-DefaultStepmx", 10.0 );
        setAsNumericalScalar( "TNC-DefaultAccuracy", 1.0e-4 );
        setAsNumericalScalar( "TNC-DefaultFmin", 1.0 );
        setAsNumericalScalar( "TNC-DefaultRescale", 1.3 );

        // OT::Base::Optim::AbdoRackwitz::AbdoRackwitz parameters //
        setAsNumericalScalar( "AbdoRackwitz-DefaultTau", 0.5 );
        setAsNumericalScalar( "AbdoRackwitz-DefaultOmega", 1.0e-4 );
        setAsNumericalScalar( "AbdoRackwitz-DefaultSmooth", 1.2 );

        // OT::Base::Optim::NearestPointAlgorithmImplementation parameters //
        setAsNumericalScalar( "NearestPointAlgorithmImplementation-DefaultLevelValue", 0.0 );
        setAsUnsignedLong( "NearestPointAlgorithmImplementation-DefaultMaximumIterationsNumber", 100 );
        setAsNumericalScalar( "NearestPointAlgorithmImplementation-DefaultMaximumAbsoluteError", 1.0e-5 );
        setAsNumericalScalar( "NearestPointAlgorithmImplementation-DefaultMaximumRelativeError", 1.0e-5 );
        setAsNumericalScalar( "NearestPointAlgorithmImplementation-DefaultMaximumResidualError", 1.0e-5 );
        setAsNumericalScalar( "NearestPointAlgorithmImplementation-DefaultMaximumConstraintError", 1.0e-5 );

        // OT::Base::Optim::Cobyla::Cobyla parameters //
        setAsNumericalScalar( "Cobyla-DefaultRhoBeg", 0.1 );

        // OT::Base::Solver::SolverImplementation parameters //
        setAsNumericalScalar( "SolverImplementation-DefaultAbsoluteError",  1e-5 );
        setAsNumericalScalar( "SolverImplementation-DefaultRelativeError",  1e-5 );
        setAsUnsignedLong( "SolverImplementation-DefaultMaximumFunctionEvaluation", 100 );

        // OT::Base::Stat::LinearModelFactory parameters //
        setAsNumericalScalar( "LinearModelFactory-DefaultLevelValue", 0.95 );

        // OT::Base::Stat::Last parameters //
        setAsUnsignedLong( "Last-DefaultMaximumSize", 65536 );

        // OT::Base::Stat::Compact parameters //
        setAsUnsignedLong( "Compact-DefaultHalfMaximumSize", 1024 );

        // OT::Base::Stat::CorrelationAnalysis parameters //
        setAsUnsignedLong( "CorrelationAnalysis-DefaultMaximumOrder", 0 );
        setAsBool( "CorrelationAnalysis-DefaultTotalIndiceComputation", false );

        // OT::Base::Stat::HaltonSequence parameters //
        setAsUnsignedLong( "HaltonSequence-InitialSeed", 1 );

        // OT::Base::Stat::SobolSequence parameters //
        setAsUnsignedLong( "SobolSequence-InitialSeed", 1 );

        // OT::Base::Stat::SensitivityAnalysis parameters //
        setAsUnsignedLong( "SensitivityAnalysis-DefaultBlockSize", 1 );
        
        // OT::Base::Stat::RandomGenerator parameters //
        setAsUnsignedLong( "RandomGenerator-InitialSeed", 0 );

        // OT::Base::Type::Collection parameters
        setAsUnsignedLong( "Collection-size-visible-in-str-from", 10 );

        // OT::Base::Type::Matrix parameters
        setAsUnsignedLong( "Matrix-size-visible-in-str-from", 5 );

        // OT::Base::Type::Tensor parameters
        setAsUnsignedLong( "Tensor-size-visible-in-str-from", 5 );

        // OT::Base::Type::MatrixImplementation parameters //
        setAsNumericalScalar( "MatrixImplementation-DefaultSmallPivot", 1.0e-7 );

        // OT::Uncertainty::Distribution::BurrFactory parameters //
        setAsNumericalScalar( "BurrFactory-AbsolutePrecision", 1e-12 );
        setAsNumericalScalar( "BurrFactory-RelativePrecision", 1e-12 );
        setAsUnsignedLong( "BurrFactory-MaximumIteration", 10 );

        // OT::Uncertainty::Distribution::DirichletFactory parameters //
        setAsUnsignedLong( "Dirichlet-DefaultSamplingSize", 500000 );
        setAsUnsignedLong( "Dirichlet-DefaultIntegrationSize", 50 );

        // OT::Uncertainty::Distribution::DirichletFactory parameters //
        setAsNumericalScalar( "DirichletFactory-ParametersEpsilon", 1e-12 );
        setAsUnsignedLong( "DirichletFactory-MaximumIteration", 10 );

        // OT::Uncertainty::Distribution::KernelSmoothing parameters //
        setAsUnsignedLong( "KernelSmoothing-SmallSize", 250 );
        setAsNumericalScalar( "KernelSmoothing-CuttOffPlugin", 5.0 );

        // OT::Uncertainty::Distribution::Normal parameters //
        setAsUnsignedLong( "Normal-MaximumNumberOfPoints", 10000000 );
        setAsUnsignedLong( "Normal-MinimumNumberOfPoints", 1000000 );
        setAsNumericalScalar( "Normal-MaximumCDFEpsilon", 5.0e-6 );
        setAsNumericalScalar( "Normal-MinimumCDFEpsilon", 5.0e-2 );

        // OT::Uncertainty::Distribution::TrapezoidalFactory parameters //
        setAsNumericalScalar( "TrapezoidalFactory-RhoBeg", 0.1 );
        setAsNumericalScalar( "TrapezoidalFactory-RhoEnd", 1.0e-5 );
        setAsUnsignedLong( "TrapezoidalFactory-MaximumIterationsNumber", 100000 );

        // OT::Uncertainty::Distribution::TruncatedDistribution parameters //
        setAsNumericalScalar( "TruncatedDistribution-DefaultThresholdRealization", 0.5 );

        // OT::Uncertainty::Distribution::TruncatedNormalFactory parameters //
        setAsNumericalScalar( "TruncatedNormalFactory-RhoBeg", 0.1 );
        setAsNumericalScalar( "TruncatedNormalFactory-RhoEnd", 1.0e-5 );
        setAsUnsignedLong( "TruncatedNormalFactory-MaximumIterationsNumber", 100000 );

        // OT::Uncertainty::Distribution::Student parameters //
        setAsUnsignedLong( "Student-MaximumNumberOfPoints", 10000000 );
        setAsUnsignedLong( "Student-MinimumNumberOfPoints", 1000000 );
        setAsNumericalScalar( "Student-MaximumCDFEpsilon", 5.0e-6 );
        setAsNumericalScalar( "Student-MinimumCDFEpsilon", 5.0e-2 );

        // OT::Uncertainty::Distribution::FrankCopulaFactory parameters //
        setAsNumericalScalar( "FrankCopulaFactory-ThetaEpsilon", 1e-14 );

        // OT::Uncertainty::Distribution::RandomMixture parameters //
        setAsUnsignedLong( "RandomMixture-DefaultBlockMin", 3 );
        setAsUnsignedLong( "RandomMixture-DefaultBlockMax", 16 );
        setAsUnsignedLong( "RandomMixture-DefaultMaxSize", 65536 );
        setAsNumericalScalar( "RandomMixture-DefaultAlpha", 5.0 );
        setAsNumericalScalar( "RandomMixture-DefaultPDFEpsilon", 1.0e-10 );
        setAsNumericalScalar( "RandomMixture-DefaultCDFEpsilon", 1.0e-10 );
        setAsNumericalScalar( "RandomMixture-GraphPDFEpsilon", 1.0e-5 );
        setAsNumericalScalar( "RandomMixture-GraphCDFEpsilon", 1.0e-5 );
        setAsUnsignedLong( "RandomMixture-SmallSize", 100 );
        setAsUnsignedLong( "RandomMixture-DefaultSizeProjection", 10000 );

        // OT::Uncertainty::Distribution::ExtraFunc::DistFunc parameters //
        setAsNumericalScalar( "DistFunc-Precision", 1.0e-14 );
        setAsUnsignedLong( "DistFunc-MaximumIteration", 5000 );

        // OT::Uncertainty::Algorithm::Simulation::DirectionalSampling::RootStrategyImplementation parameters //
        setAsNumericalScalar( "RootStrategyImplementation-DefaultMaximumDistance", 8.0 );
        setAsNumericalScalar( "RootStrategyImplementation-DefaultStepSize", 1.0 );

        // OT::Uncertainty::Algorithm::Simulation::Simulation parameters //
        setAsUnsignedLong( "Simulation-DefaultMaximumOuterSampling", 1000 );
        setAsNumericalScalar( "Simulation-DefaultMaximumCoefficientOfVariation", 1.0e-1 );
        setAsNumericalScalar( "Simulation-DefaultMaximumStandardDeviation", 0.0 );
        setAsUnsignedLong( "Simulation-DefaultBlockSize", 1 );

        // OT::Uncertainty::Algorithm::Simulation::SimulationResultImplementation parameters //
        setAsNumericalScalar( "SimulationResultImplementation-DefaultConfidenceLevel", 0.95 );

        // OT::Uncertainty::Algorithm::Analytical::AnalyticalResult parameters //
        setAsNumericalScalar( "AnalyticalResult-DefaultWidth", 1.0 );

        // OT::Uncertainty::Algorithm::Analytical::StrongMaximumTest parameters //
        setAsNumericalScalar( "StrongMaximumTest-DefaultDeltaPrecision", 1.0e-7 );
        setAsNumericalScalar( "StrongMaximumTest-Epsilon", 1.0e-10 );

        // OT::Uncertainty::Algorithm::MetaModel::FunctionalChaos::CleaningStrategy parameters //
        setAsUnsignedLong( "CleaningStrategy-DefaultMaximumSize", 20 );
        setAsNumericalScalar( "CleaningStrategy-DefaultSignificanceFactor", 1.0e-4 );

        // OT::Uncertainty::Algorithm::MetaModel::FunctionalChaos::FunctionalChaosAlgorithm parameters //
        setAsNumericalScalar( "FunctionalChaosAlgorithm-DefaultMaximumResidual", 1.0e-6 );

        // OT::Uncertainty::Algorithm::Experiments::WeightedExperiment parameters //
        setAsUnsignedLong( "WeightedExperiment-DefaultSize", 100 );

        // OT::Uncertainty::Algorithm::Experiments::GaussProductExperiment parameters //
        setAsUnsignedLong( "GaussProductExperiment-DefaultMarginalDegree", 5 );

        // OT::Uncertainty::Algorithm::OrthogonalBasis::HyperbolicAnisotropicEnumerateFunction parameters //
        setAsNumericalScalar( "HyperbolicAnisotropicEnumerateFunction-DefaultQ", 0.4 );

        // OT::Uncertainty::Algorithm::IsoProbabilisticTransformation::MarginalTransformation::MarginalTransformationEvaluation parameters //
        setAsNumericalScalar( "MarginalTransformationEvaluation-DefaultTailThreshold", 0.99 );

        // OT::Uncertainty::Model::DistributionImplementation parameters //
        setAsUnsignedLong( "DistributionImplementation-DefaultPointNumber", 129 );
        setAsNumericalScalar( "DistributionImplementation-DefaultQuantileEpsilon", 1.0e-12 );
        setAsNumericalScalar( "DistributionImplementation-DefaultPDFEpsilon", 1.0e-14 );
        setAsNumericalScalar( "DistributionImplementation-DefaultCDFEpsilon", 1.0e-14 );
        setAsUnsignedLong( "DistributionImplementation-DefaultQuantileIteration", 10 );
        setAsNumericalScalar( "DistributionImplementation-QMin", 0.15 );
        setAsNumericalScalar( "DistributionImplementation-QMax", 0.85 );
        setAsUnsignedLong( "DistributionImplementation-DefaultIntegrationNodesNumber", 255 );
        setAsUnsignedLong( "DistributionImplementation-DefaultLevelNumber", 10 );
        setAsUnsignedLong( "DistributionImplementation-DefaultQuantileCacheSize", 128 );

        // OT::Uncertainty::Model::ContinuousDistribution parameters //
        setAsUnsignedLong( "ContinuousDistribution-DefaultIntegrationNodesNumber", 51 );

        // OT::Uncertainty::Model::DiscreteDistribution parameters //
        setAsNumericalScalar( "DiscreteDistribution-SupportEpsilon", 1.0e-14 );

        // OT::Uncertainty::Model::DistributionImplementationFactory parameters //
        setAsUnsignedLong( "DistributionImplementationFactory-DefaultBootstrapSize", 100 );

        // OT::Uncertainty::StatTests::VisualTest parameters //
        setAsUnsignedLong( "VisualTest-KendallPlot-MonteCarloSize", 100 );
      }

      /* String converter */
      String ResourceMap::__repr__() const
      {
        OSS oss;
        oss << "ResourceMap={";
        const char * separator = "\n  ";
        for(MapType::const_iterator it = map_.begin(); it != map_.end(); it++, separator=",\n  ")
          oss << separator
              << it->first << " => " << it->second;
        oss << "\n}";
        return oss;
      }



      /*
       * @fn std::ostream & operator <<(std::ostream & os, const ResourceMap & obj)
       * @brief Output stream converter
       * @param os A STL output stream resourceMap
       * @param obj The resourceMap read by \em os
       * @return A reference to \em os
       *
       * Operator << converts the ResourceMap object to an output stream
       * so it is easy to show the content of the resourceMap.
       */
      std::ostream & operator <<(std::ostream & os, const ResourceMapInstance & obj)
      {
        return os << obj.lock().__repr__();
      }

      OStream & operator <<(OStream & OS, const ResourceMapInstance & obj)
      {
        return OS << obj.lock().__repr__();
      }





    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
