//                                               -*- C++ -*-
/**
 * @file  PythonNumericalMathEvaluationImplementation.cxx
 * @brief This class binds a Python function to an Open TURNS' NumericalMathFunction
 *
 * (C) Copyright 2005-2010 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 */

#include "PythonNumericalMathEvaluationImplementation.hxx"
#include "OSS.hxx"
#include "Description.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(PythonNumericalMathEvaluationImplementation);

      /* Default constructor */
      PythonNumericalMathEvaluationImplementation::PythonNumericalMathEvaluationImplementation(PyObject * pyCallable)
	: NumericalMathEvaluationImplementation(),
	  pyObj_(pyCallable)
      {
	Py_XINCREF( pyCallable );
	// One MUST initialize the description with the correct dimension
	const UnsignedLong inputDimension(getInputDimension());
	const UnsignedLong outputDimension(getOutputDimension());
	Description description(inputDimension + outputDimension);
	for (UnsignedLong i = 0; i < inputDimension; ++i) description[i] = (OSS() << "x" << i);
	for (UnsignedLong i = 0; i < outputDimension; ++i) description[inputDimension + i] = (OSS() << "y" << i);
	setDescription(description);
      }

      /* Virtual constructor */
      PythonNumericalMathEvaluationImplementation * PythonNumericalMathEvaluationImplementation::clone() const
      {
	return new PythonNumericalMathEvaluationImplementation(*this);
      }

      /* Copy constructor */
      PythonNumericalMathEvaluationImplementation::PythonNumericalMathEvaluationImplementation(const PythonNumericalMathEvaluationImplementation & other)
	: NumericalMathEvaluationImplementation(other),
	  pyObj_(other.pyObj_)
      {
	Py_XINCREF( pyObj_ );
      }

      /* Destructor */
      PythonNumericalMathEvaluationImplementation::~PythonNumericalMathEvaluationImplementation()
      {
	Py_XDECREF( pyObj_ );
      }

      /* Comparison operator */
      Bool PythonNumericalMathEvaluationImplementation::operator ==(const PythonNumericalMathEvaluationImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String PythonNumericalMathEvaluationImplementation::str() const {
	OSS oss;
	oss << "class=" << PythonNumericalMathEvaluationImplementation::GetClassName()
	    << " name=" << getName();
	return oss;
      }
  
      /* Test for actual implementation */
      Bool PythonNumericalMathEvaluationImplementation::isActualImplementation() const
      {
	return true;
      }






      /* Here is the interface that all derived class must implement */
	
      /* Operator () */
      PythonNumericalMathEvaluationImplementation::NumericalPoint PythonNumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
      /*	throw(InvalidArgumentException,InternalException)*/
      {
	++callsNumber_;
	const UnsignedLong size(in.getSize());

	PyObject * inTuple(PyTuple_New( size ));

	for (UnsignedLong i = 0; i < size; ++i) PyTuple_SetItem( inTuple, i, PyFloat_FromDouble( in[i] ) );

	PyObject * result = PyObject_CallFunction( pyObj_,
						   const_cast<char *>( "(O)" ),
						   inTuple );

	NumericalPoint out;

	if ( PySequence_Check( result ) )
	  {
	    const long length = PySequence_Size( result );
	    out               = NumericalPoint( length );
	    for (long i = 0; i < length; ++i) out[i] = PyFloat_AsDouble( PySequence_GetItem( result, i ) );
	  }
	
	Py_XDECREF( inTuple );
	Py_XDECREF( result  );
	
	return out;
      }

      /*      PythonNumericalMathEvaluationImplementation::NumericalSample PythonNumericalMathEvaluationImplementation::operator() (const NumericalSample & in) const
	throw(InvalidArgumentException,InternalException)
      {
	const UnsignedLong size(in.getSize());
	callsNumber_ += size;
	if (size == 0) throw InvalidArgumentException(HERE) << "Error: trying to evaluate a function over an empty sample.";
	const UnsignedLong dimension(in.getDimension());

	PyObject * inTuple(PyTuple_New( size ));
	PyObject * inPoints[size];
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    inPoints[i] = PyTuple_New( dimension );
	    for (UnsignedLong j = 0; j < dimension; ++j) PyTuple_SetItem( inPoints[i], j, PyFloat_FromDouble( in[i][j] ) );
	    PyTuple_SetItem( inTuple, i, inPoints[i] );
	    //	    Py_XDECREF( inPoints[i] );
	  }
	std::cout << "before call" << std::endl;
	PyObject * result = PyObject_CallFunction( pyObj_,
						   const_cast<char *>( "(O)" ),
						   inTuple );
	std::cout << "after call" << std::endl;
	NumericalSample out;

	if ( PySequence_Check( result ) )
	  {
	    const long length = PySequence_Size( result );
	    if ( (length == size) && PySequence_Check( PySequence_GetItem( result, 0 )) )
	      {
		const long dim = PySequence_Size( PySequence_GetItem( result, 0 ));
		out            = NumericalSample( length, dim );
		for (long i = 0; i < length; ++i)
		  for (long j = 0; j < dim; ++j)
		    out[i][j] = PyFloat_AsDouble( PySequence_GetItem(PySequence_GetItem( result, i ), j) );
	      }
	  }
	
	Py_XDECREF( inTuple );
	Py_XDECREF( result  );
	
	return out;
	} */

      /* Accessor for input point dimension */
      UnsignedLong PythonNumericalMathEvaluationImplementation::getInputDimension() const
      /*	throw(InternalException)*/
      {
	PyObject * result = PyObject_CallMethod( pyObj_,
						 const_cast<char *>( "getInputDimension" ),
						 const_cast<char *>( "()"                              ) );
	UnsignedLong dim = PyLong_AsLong( result );
	Py_DECREF( result );
	return dim;
      }
      
      /* Accessor for output point dimension */
      UnsignedLong PythonNumericalMathEvaluationImplementation::getOutputDimension() const
      /*	throw(InternalException)*/
      {
	PyObject * result = PyObject_CallMethod( pyObj_,
						 const_cast<char *>( "getOutputDimension" ), 
						 const_cast<char *>( "()"                               ) );
	UnsignedLong dim = PyLong_AsLong( result );
	Py_DECREF( result );
	return dim;
      }
      
    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
