//                                               -*- C++ -*-
/**
 *  @file  t_FunctionalChaos_ishigami.cxx
 *  @brief The test file of MonteCarlo class
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (mer. 21 mai 2008) $
 *  Id:      $Id: t_FunctionalChaos_ishigami.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include <iostream>
#include <iomanip>
#include <cmath>
#include "OT.hxx"
#include "OSS.hxx"
#include "OTtestcode.hxx"
#include "OStream.hxx"
#include "FunctionalChaosAlgorithm.hxx"
#include "FunctionalChaosResult.hxx"
#include "FunctionalChaosRandomVector.hxx"
#include "LHSExperiment.hxx"
#include "MonteCarloExperiment.hxx"
#include "LowDiscrepancyExperiment.hxx"
#include "LeastSquaresStrategy.hxx"
#include "ProjectionStrategy.hxx"
#include "AdaptiveStrategy.hxx"
#include "CleaningStrategy.hxx"
#include "FixedStrategy.hxx"
#include "SequentialStrategy.hxx"
#include "NumericalPoint.hxx"
#include "Description.hxx"
#include "NumericalMathFunction.hxx"
#include "Collection.hxx"
#include "ComposedDistribution.hxx"
#include "Distribution.hxx"
#include "EnumerateFunction.hxx"
#include "Indices.hxx"
#include "OrthogonalProductPolynomialFactory.hxx"
#include "OrthogonalUniVariatePolynomialFamily.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Func;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;
using namespace OT::Uncertainty::Algorithm;

NumericalScalar sobol(const Indices & indices,
		      const NumericalPoint & a)
{
  NumericalScalar value(1.0);
  for (UnsignedLong i = 0; i < indices.getSize(); ++i)
    {
      value *= 1.0 / (3.0 * pow(1.0 + a[indices[i]], 2.0));
    }
  return value;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  OStream fullprint(std::cout);
  setRandomGenerator();

  try {

    // Problem parameters
    UnsignedLong dimension(5);

    // Reference analytical values
    NumericalScalar meanTh(1.0);
    NumericalScalar covTh(1.0);
    NumericalPoint a(dimension);
    // Create the Ishigami function
    Description inputVariables(dimension);
    Description outputVariables(1);
    outputVariables[0] = "y";
    Description formula(1);
    formula[0] = "1.0";
    for (UnsignedLong i = 0; i < dimension; ++i)
      {
	a[i] = 0.5 * i;
	covTh *= 1.0 + 1.0 / (3.0 * pow(1.0 + a[i], 2.0));
	inputVariables[i] = (OSS() << "xi" << i);
	formula[0] = (OSS() << formula[0] << " * ((abs(4.0 * xi" << i << " - 2.0) + " << a[i] << ") / (1.0 + " << a[i] << "))");
      }
    --covTh;

    NumericalMathFunction model(inputVariables, outputVariables, formula);

    // Create the input distribution
    Collection<Distribution> marginals(dimension);
    for (UnsignedLong i = 0; i < dimension; ++i)
      {
	marginals[i] = Uniform(0.0, 1.0);
      }
    ComposedDistribution distribution(marginals);

    // Create the orthogonal basis
    Collection<OrthogonalUniVariatePolynomialFamily> polynomialCollection(dimension);
    for (UnsignedLong i = 0; i < dimension; ++i)
      {
	polynomialCollection[i] = LegendreFactory();
      }
    EnumerateFunction enumerateFunction(dimension);
    OrthogonalProductPolynomialFactory productBasis(polynomialCollection, enumerateFunction);

    // Create the adaptive strategy
    // We can choose amongst several strategies
    // First, the most efficient (but more complex!) strategy
    Collection<AdaptiveStrategy> listAdaptiveStrategy(0);
    UnsignedLong indexMax(200);
    UnsignedLong basisDimension(20);
    NumericalScalar threshold(1.0e-6);
    listAdaptiveStrategy.add(CleaningStrategy(productBasis, indexMax, basisDimension, threshold, true));
    // Second, the most used (and most basic!) strategy
    UnsignedLong degree(6);
    listAdaptiveStrategy.add(FixedStrategy(productBasis, enumerateFunction.getStrateCumulatedCardinal(degree)));
    // Third, a slight enhancement with respect to the basic strategy
    degree = 3;
    listAdaptiveStrategy.add(SequentialStrategy(productBasis, enumerateFunction.getStrateCumulatedCardinal(degree), true));

    for(UnsignedLong adaptiveStrategyIndex = 0; adaptiveStrategyIndex < listAdaptiveStrategy.getSize(); ++adaptiveStrategyIndex)
      {
        AdaptiveStrategy adaptiveStrategy(listAdaptiveStrategy[adaptiveStrategyIndex]);
        // Create the projection strategy
        UnsignedLong samplingSize(250);
        Collection<ProjectionStrategy> listProjectionStrategy(0);
        // We have only the LeastSquaresStrategy up to now (0.13.0) but we can choose several sampling schemes
        // Monte Carlo sampling
        listProjectionStrategy.add(LeastSquaresStrategy(MonteCarloExperiment(samplingSize)));
        // LHS sampling
        listProjectionStrategy.add(LeastSquaresStrategy(LHSExperiment(samplingSize)));
        // Low Discrepancy sequence
        listProjectionStrategy.add(LeastSquaresStrategy(LowDiscrepancyExperiment(LowDiscrepancySequence(SobolSequence()),samplingSize)));
        for(UnsignedLong projectionStrategyIndex = 0; projectionStrategyIndex < listProjectionStrategy.getSize(); ++projectionStrategyIndex)
          {
            ProjectionStrategy projectionStrategy(listProjectionStrategy[projectionStrategyIndex]);
            // Create the polynomial chaos algorithm
            NumericalScalar maximumResidual(1.0e-10);
            FunctionalChaosAlgorithm algo(model, distribution, adaptiveStrategy, projectionStrategy);
            algo.setMaximumResidual(maximumResidual);
            // Reinitialize the RandomGenerator to see the effect of the sampling method only
	    RandomGenerator::SetSeed(0);
            algo.run();

            // Examine the results
            FunctionalChaosResult result(algo.getResult());
            fullprint << "//////////////////////////////////////////////////////////////////////" << std::endl;
            fullprint << adaptiveStrategy << std::endl;
            fullprint << projectionStrategy << std::endl;
            NumericalScalar residual(result.getResidual());
            fullprint << "residual=" << std::fixed << std::setprecision(5) << residual << std::endl;

            // Post-process the results
            FunctionalChaosRandomVector vector(result);
            NumericalScalar mean(vector.getMean()[0]);
            fullprint << "mean=" << std::fixed << std::setprecision(5) << mean << " absolute error=" << std::scientific << std::setprecision(1) << fabs(mean - meanTh) << std::endl;
            NumericalScalar variance(vector.getCovariance()(0, 0));
            fullprint << "variance=" << std::fixed << std::setprecision(5) << variance << " absolute error=" << std::scientific << std::setprecision(1) << fabs(variance - covTh) << std::endl;
	    Indices indices(1);
            for(UnsignedLong i = 0; i < dimension; ++i)
              {
		indices[0] = i;
                NumericalScalar value(vector.getSobolIndex(i));
                fullprint << "Sobol index " << i << " = " << std::fixed << std::setprecision(5) << value << " absolute error=" << std::scientific << std::setprecision(1) << fabs(value - sobol(indices, a)) << std::endl;
              }
            indices = Indices(2);
            UnsignedLong k(0);
            for (UnsignedLong i = 0; i < dimension; ++i)
              {
                indices[0] = i;
                for (UnsignedLong j = i+1; j < dimension; ++j)
                  {
                    indices[1] = j;
                    NumericalScalar value(vector.getSobolIndex(indices));
                    fullprint << "Sobol index " << indices << " =" << std::fixed << std::setprecision(5) << value << " absolute error=" << std::scientific << std::setprecision(1) << fabs(value - sobol(indices, a)) << std::endl;
                    k = k+1;
                  }
              }
            indices = Indices(3);
            indices[0] = 0;
            indices[1] = 1;
            indices[2] = 2;
            NumericalScalar value(vector.getSobolIndex(indices));
            fullprint << "Sobol index " << indices << " =" << std::fixed << std::setprecision(5) << value << " absolute error=" << std::scientific << std::setprecision(1) << fabs(value - sobol(indices, a)) << std::endl;
          }
      }
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }
  
  
  return ExitCode::Success;
}
