//                                               -*- C++ -*-
/**
 *  @file  MultiNomial.cxx
 *  @brief The MultiNomial distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: MultiNomial.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "Collection.hxx"
#include "MultiNomial.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "Exception.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Stat::RandomGenerator              RandomGenerator;
      typedef Base::Type::Collection<UnsignedLong>     UnsignedLongCollection;

      CLASSNAMEINIT(MultiNomial);

      static Base::Common::Factory<MultiNomial> RegisteredFactory("MultiNomial");

      /* Default constructor */
      MultiNomial::MultiNomial()
	/* throw (InvalidArgumentException) */
	: DiscreteDistribution("MultiNomial"),
	  p_(1, 0.5),
	  n_(2)
      {
	// We set the dimension of the MultiNomial distribution
	setDimension( p_.getDimension() );
	setP( p_ );
      }

      /* Parameters constructor */
      MultiNomial::MultiNomial(const NumericalPoint & p, const UnsignedLong n)
	/* throw (InvalidArgumentException) */
	: DiscreteDistribution("MultiNomial"),
	  p_(p.getDimension()), n_(n)
      {
	// We set the dimension of the MultiNomial distribution
	setDimension( p.getDimension() );
	setP( p );
      }

      /* Comparison operator */
      Bool MultiNomial::operator ==(const MultiNomial & other) const
      {
	if (this != &other) { // Other is NOT me, so I have to make the comparison
	  if (n_ != other.n_) return false;
	  const UnsignedLong dimension(getDimension());
	  for (UnsignedLong i = 0; i < dimension; ++i)
	    {
	      if (p_[i] != other.p_[i]) return false;
	    }
	  return true;
	} else return true;
      }
  
      /* String converter */
      String MultiNomial::__repr__() const
      {
	OSS oss;
	oss << "class=" << MultiNomial::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " p=" << p_
	    << " n=" << n_;
	return oss;
      }
  
      /* Virtual constructor */
      MultiNomial * MultiNomial::clone() const
      {
	return new MultiNomial(*this);
      }

      /* Get one realization of the distribution */
      MultiNomial::NumericalPoint MultiNomial::getRealization() const
      {
	UnsignedLong N(n_);
	const UnsignedLong dimension(getDimension());
	NumericalPoint realization(dimension);
	/* We use an elementary algorithm based on the definition of the MultiNomial distribution:
	 * the i-th (i in 2..dim-) component is thrown according to the binomial distribution with
	 * parameters conditioned by the value taken by the (i-1)th preceding components, then the
	 * first component takes the remaining value to sum to n. */
	for (UnsignedLong i = 1; i < dimension; ++i)
	  {
	    /* The current component follow a binomial distribution with parameters p_[i] and N */
	    /* Generation of a binomial realization with parameters p_[i] and N as a sum of Bernoulli
               random variables */
	    /* Choose the most efficient case between throwing success or fail */
	    const NumericalScalar pI(std::min(p_[i], 1.0 - p_[i]));
	    /* Probability of one fail */
	    const NumericalScalar qI(1.0 - pI);
	    /* Probability of N fails */
	    const NumericalScalar qIPowN(pow(qI, N));
	    /* xI will be the realization of the binomial. We use CDF inversion. */
	    UnsignedLong xI(0);
	    const NumericalScalar remainder(pI / qI);
	    const NumericalScalar factor(remainder * (N + 1.0));
	    NumericalScalar u(RandomGenerator::Generate());
	    NumericalScalar pdf(qIPowN);
	    while (u >= pdf)
	      {
		/* Another success */
		++xI;
		/* Update the remaining probability... */
		u -= pdf;
		/* ... and the probability of failure for the next candidate value xI */
		pdf *= factor / xI - remainder;
	      }
	    /* If we used the number of fails, take the tail number of success */
	    if (p_[i] != pI) xI = N - xI;
	    realization[i] = xI;
	    N -= xI;
	  }
	realization[0] = N;
	return realization;
      }
     

      /* Get the PDF of the distribution */
      NumericalScalar MultiNomial::computePDF(const NumericalPoint & point) const
      {
	const UnsignedLong dimension(getDimension());
	if (point.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the given point has a dimension not compatible with the distribution dimension";
	// First, check the validity of the input
	NumericalScalar sumX(0.0);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    const NumericalScalar k(point[i]);
	    if ((fabs(k - round(k)) > DiscreteDistribution::SupportEpsilon) || (k < -DiscreteDistribution::SupportEpsilon) || (k > n_ + DiscreteDistribution::SupportEpsilon)) return 0.0;
	    sumX += point[i];
	  }
	if (fabs(sumX - n_) > DiscreteDistribution::SupportEpsilon) return 0.0;
	NumericalScalar pdf(SpecFunc::LnGamma(n_ + 1.0));
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    const NumericalScalar k(point[i]);
	    pdf += k * log(p_[i]) - SpecFunc::LnGamma(k + 1.0);
	  }
	return exp(pdf);
      }


      /* Get the CDF of the distribution */
      NumericalScalar MultiNomial::computeCDF(const NumericalPoint & point,
					      const Bool tail) const
      {
	// FIXME
	const UnsignedLong dimension(getDimension());
	if (point.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the given point has a dimension not compatible with the distribution dimension";
	NumericalScalar cdf(0.0);
	UnsignedLongCollection integerPoint(dimension);
	// First, check the bording cases
	Bool outside(true);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    // If the given point does not cover any point of the support, return 0.0
	    if (point[i] < -DiscreteDistribution::SupportEpsilon) return 0.0;
	    // We must check that point[i] >= 0.0 before the conversion to an UnsignedLong
	    integerPoint[i] = UnsignedLong(floor(point[i] + DiscreteDistribution::SupportEpsilon));
	    outside &= (integerPoint[i] >= n_);
	  }
	// If the point covers the whole support of the distribution, return 1.0
	if (outside) return 1.0;
	/* Size of the sample to be generated: levels[0] * ... * levels[dimension-1] */
	UnsignedLong size(integerPoint[0]);
	for (UnsignedLong i = 1; i < dimension; ++i) size *= integerPoint[i];
	NumericalSample boxPlane(size, dimension);
	/* Indices would have stored the indices of the nested loops if we were able to code "dimension" nested loops dynamically */
	UnsignedLongCollection indices(dimension, 0);
	for (UnsignedLong flatIndex = 0; flatIndex < size; ++flatIndex)
	  {
	    NumericalPoint currentPoint(dimension);
	    for (UnsignedLong i = 0; i < dimension - 1; ++i) currentPoint[i] = indices[i];
	    cdf += computeCDF(currentPoint);
	    /* Update the indices */
	    ++indices[0];
	    /* Propagate the remainders */
	    for (UnsignedLong i = 0; i < dimension - 1; ++i) indices[i + 1] += (indices[i] == integerPoint[i]);
	    /* Correction of the indices. The last index cannot overflow. */
	    for (UnsignedLong i = 0; i < dimension - 1; ++i) indices[i] = indices[i] % integerPoint[i];
	  }
	return cdf;
      }

      /** Get the PDFGradient of the distribution */
      MultiNomial::NumericalPoint MultiNomial::computePDFGradient(const NumericalPoint & point) const
      {
	const UnsignedLong dimension(getDimension());
	if (point.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the given point has a dimension not compatible with the distribution dimension";
	const NumericalScalar pdf(computePDF(point));
	NumericalPoint pdfGradient(dimension);
	if (pdf > 0.0)
	  {
	    for (UnsignedLong i = 0; i < dimension; ++i)
	      {
		pdfGradient[i] = pdf * point[i] / p_[i];
	      }
	  }
	return pdfGradient;
      }

      /** Get the CDFGradient of the distribution */
      MultiNomial::NumericalPoint MultiNomial::computeCDFGradient(const NumericalPoint & point) const
      {
	const UnsignedLong dimension(getDimension());
	if (point.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the given point has a dimension not compatible with the distribution dimension";
	// const NumericalScalar cdf(computeCDF(point));
	// To be implemented
	throw NotYetImplementedException(HERE);
      }

      /* Get the quantile of the distribution */
      MultiNomial::NumericalPoint MultiNomial::computeQuantile(const NumericalScalar prob,
							       const Bool tail) const
      {
	if (prob <= 0.0 || prob >= 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of ]0, 1[";
	throw NotYetImplementedException(HERE);
      }

      /* Get the mean of the distribution */
      MultiNomial::NumericalPoint MultiNomial::getMean() const /* throw(NotDefinedException) */
      {
	return getDimension() * p_;
      }

      /* Get the covariance of the distribution */
      MultiNomial::CovarianceMatrix MultiNomial::getCovariance() const /* throw(NotDefinedException) */
      {
	const UnsignedLong dimension(getDimension());
	CovarianceMatrix covariance(SquareMatrix(dimension).getImplementation());
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    NumericalScalar pI(p_[i]);
	    for (UnsignedLong j = 0; j < i; ++j)
	      {
		covariance(i, j) = -dimension * pI * p_[j];
	      }
	    covariance(i, i) = dimension * pI * (1.0 - pI);
	  }
	return covariance;
      }

      /* Parameters value and description accessor */
      MultiNomial::NumericalPointWithDescriptionCollection MultiNomial::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	const UnsignedLong dimension(getDimension());
	NumericalPointWithDescription point(p_);
        Description description(dimension);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    OSS oss;
	    oss << "p_" << i;
	    description[i] = oss;
	  }
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }




      /* P accessor */
      void MultiNomial::setP(const NumericalPoint & p)
	/* throw (InvalidArgumentException) */
      {
	// We check that the elements are all positive
	UnsignedLong dimension(getDimension());
	if (p.getDimension() != dimension) throw InvalidArgumentException(HERE) << "P has incorrect dimension, dimension=" << p.getDimension() << " instead of " << dimension;
	NumericalScalar sum(0.0);
	for(UnsignedLong i = 0; i < dimension; ++i)
	  {
	    NumericalScalar pI(p[i]);
	    if (pI <= 0.0) throw InvalidArgumentException(HERE) << "P elements MUST be positive";
	    sum += pI;
	  }
	// If OK, then we renormalize the vector and we set the correct dimension.
	p_ = (1.0 / sum) * p;
      }

      /* P accessor */
      MultiNomial::NumericalPoint MultiNomial::getP() const
      {
	return p_;
      }

      /* N accessor */
      void MultiNomial::setN(const UnsignedLong n)
      {
	n_ = n;
      }

      UnsignedLong MultiNomial::getN() const
      {
	return n_;
      }




    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
