//                                               -*- C++ -*-
/**
 *  @file  Gumbel.cxx
 *  @brief The Gumbel distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Gumbel.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "Gumbel.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Gumbel);

      static Base::Common::Factory<Gumbel> RegisteredFactory("Gumbel");

      /* Default constructor */
      Gumbel::Gumbel()
        : NonEllipticalDistribution("Gumbel"),
          alpha_(1.0), beta_(0.0)
      {
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      Gumbel::Gumbel(const NumericalScalar arg1,
                     const NumericalScalar arg2,
                     const ParameterSet set)
        /* throw (InvalidArgumentException) */
        : NonEllipticalDistribution("Gumbel"),
          alpha_(0.0),
          beta_(arg2)
      {
        switch (set) {
        case ALPHABETA:
          setAlpha(arg1);
          break;

        case MUSIGMA:
          setMuSigma(arg1, arg2);
          break;

        default:
          throw InvalidArgumentException(HERE) << "Invalid parameter set argument";
        }
        setDimension( 1 );
        computeRange();
      }

      /* Comparison operator */
      Bool Gumbel::operator ==(const Gumbel & other) const {
        Bool sameObject = false;

        if (this != &other) { // Other is NOT me, so I have to realize the comparison
          // sameObject = ...
          // TODO: Write Gumbel::operator ==(...)
          if ( (alpha_ == other.alpha_) && (beta_ == other.beta_) )
            sameObject = true;
        } else sameObject = true;

        return sameObject;
      }

      /* String converter */
      String Gumbel::__repr__() const {
        OSS oss;
        oss << "class=" << Gumbel::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " alpha=" << alpha_
            << " beta=" << beta_;
        return oss;
      }

      /* Virtual constructor */
      Gumbel * Gumbel::clone() const
      {
        return new Gumbel(*this);
      }

      /* Get one realization of the distribution */
      Gumbel::NumericalPoint Gumbel::getRealization() const
      {
        return NumericalPoint(1, beta_ - log(-log(RandomGenerator::Generate())) / alpha_);
      }



      /* Get the DDF of the distribution */
      Gumbel::NumericalPoint Gumbel::computeDDF(const NumericalPoint & point) const
      {
        NumericalScalar expX(exp(-alpha_ * (point[0] - beta_)));
        return NumericalPoint(1, alpha_ * alpha_ * (expX - 1.0) * expX * exp(-expX));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Gumbel::computePDF(const NumericalPoint & point) const
      {
        NumericalScalar expX(exp(-alpha_ * (point[0] - beta_)));
        return alpha_ * expX * exp(-expX);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Gumbel::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        NumericalScalar x(-alpha_ * (point[0] - beta_));
        NumericalScalar expX(exp(x));
        if (tail)
          {
            // 17.524801012400200133966665679584855800344831597510 = numerical bound for which the approximation has a relative error less than 1e-16
            if (x < -1.94)
              {
                NumericalScalar value(expX);
                NumericalScalar coeff(1.0);
                for (UnsignedLong i = 2; i < 10; ++i)
                  {
                    coeff *= -expX / static_cast<NumericalScalar>(i);
                    value += coeff;
                  }
                return value;
              }
            return 1.0 - exp(-expX);
          }
        return exp(-expX);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Gumbel::computeCharacteristicFunction(const NumericalScalar x,
                                                              const Bool logScale) const
      {
        if (logScale) return log(SpecFunc::Gamma(NumericalComplex(1.0, -x / alpha_))) + NumericalComplex(0.0, beta_ * x);
        return SpecFunc::Gamma(NumericalComplex(1.0, -x / alpha_)) * exp(NumericalComplex(0.0, beta_ * x));
      }

      /* Get the PDFGradient of the distribution */
      Gumbel::NumericalPoint Gumbel::computePDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar x(point[0] - beta_);
        NumericalScalar expX(exp(-alpha_ * x));
        NumericalScalar pdf(alpha_ * expX * exp(-expX));
        NumericalPoint pdfGradient(2);
        pdfGradient[0] = (1.0 / alpha_ - x * (1.0 - expX)) * pdf;
        pdfGradient[1] = alpha_ * (1.0 - expX) * pdf;
        return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Gumbel::NumericalPoint Gumbel::computeCDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar x(point[0] - beta_);
        NumericalScalar expX(exp(-alpha_ * x));
        NumericalScalar cdf(exp(-expX));
        NumericalPoint cdfGradient(2);
        cdfGradient[0] = x * expX * cdf;
        cdfGradient[1] = -alpha_ * expX * cdf;
        return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar Gumbel::computeScalarQuantile(const NumericalScalar prob,
						    const Bool tail,
                                                    const NumericalScalar precision) const
      {
	if (tail) return beta_ - log(-log(1.0 - prob)) / alpha_;
        return beta_ - log(-log(prob)) / alpha_;
      }

      /* Get the mean of the distribution */
      Gumbel::NumericalPoint Gumbel::getMean() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, beta_ + SpecFunc::EulerConstant / alpha_);
      }

      /* Get the standard deviation of the distribution */
      Gumbel::NumericalPoint Gumbel::getStandardDeviation() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, getSigma());
      }

      /* Get the skewness of the distribution */
      Gumbel::NumericalPoint Gumbel::getSkewness() const /* throw(NotDefinedException) */
      {
        // 1.139547099404648657492793 = 12 * sqrt(6) * Zeta(3) / Pi^3
        return NumericalPoint(1, 1.139547099404648657492793);
      }

      /* Get the kurtosis of the distribution */
      Gumbel::NumericalPoint Gumbel::getKurtosis() const /* throw(NotDefinedException) */
      {
        // 5.4 = 27/5
        return NumericalPoint(1, 5.4);
      }

      /* Get the covariance of the distribution */
      Gumbel::CovarianceMatrix Gumbel::getCovariance() const /* throw(NotDefinedException) */
      {
        CovarianceMatrix covariance(1);
        covariance(0, 0) = SpecFunc::PI2_6 / (alpha_ * alpha_);
        return covariance;
      }

      /* Parameters value and description accessor */
      Gumbel::NumericalPointWithDescriptionCollection Gumbel::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(2);
        Description description(point.getDimension());
        point[0] = alpha_;
        point[1] = beta_;
        description[0] = "alpha";
        description[1] = "beta";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void Gumbel::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = Gumbel(parametersCollection[0][0], parametersCollection[0][1]);
      }





      /* Alpha accessor */
      void Gumbel::setAlpha(const NumericalScalar alpha)
        /* throw(InvalidArgumentException) */
      {
        if (alpha <= 0.) throw InvalidArgumentException(HERE) << "Alpha MUST be positive";
        alpha_ = alpha;
        computeRange();
      }

      NumericalScalar Gumbel::getAlpha() const
      {
        return alpha_;
      }


      /* M accessor */
      void Gumbel::setBeta(const NumericalScalar beta)
      {
        beta_ = beta;
        computeRange();
      }

      NumericalScalar Gumbel::getBeta() const
      {
        return beta_;
      }

      /* Mu accessor */
      void Gumbel::setMuSigma(const NumericalScalar mu,
                              const NumericalScalar sigma)
        /* throw(InvalidArgumentException) */
      {
        if (sigma <= 0.0) throw InvalidArgumentException(HERE) << "Sigma must be > 0, here sigma=" << sigma;
        alpha_ = SpecFunc::PI_SQRT6 / sigma;
        beta_ = mu - SpecFunc::EULERSQRT6_PI * sigma;
        computeRange();
      }

      NumericalScalar Gumbel::getMu() const
      {
        return beta_ + SpecFunc::EulerConstant / alpha_;
      }

      NumericalScalar Gumbel::getSigma() const
      {
        return SpecFunc::PI_SQRT6 / alpha_;
      }

      /* Method save() stores the object through the StorageManager */
      void Gumbel::save(StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
        adv.saveAttribute( "alpha_", alpha_ );
        adv.saveAttribute( "beta_", beta_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void Gumbel::load(StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);
        adv.loadAttribute( "alpha_", alpha_ );
        adv.loadAttribute( "beta_", beta_ );
        computeRange();
      }



    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
