//                                               -*- C++ -*-
/**
 *  @file  NumericalPoint.cxx
 *  @brief NumericalPoint implements the classical mathematical point
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: NumericalPoint.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "NumericalPoint.hxx"
#include "Exception.hxx"
#include "StorageManager.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      typedef Common::OutOfBoundException OutOfBoundException;

      CLASSNAMEINIT(NumericalPoint);

      static Common::Factory<NumericalPoint> RegisteredFactory("NumericalPoint");

      /* Default constructor */
      NumericalPoint::NumericalPoint()
	: PersistentCollection<NumericalScalar>() //,
	  // p_description_()
      {
        // Nothing to do
      }

      /* Constructor with size */
      NumericalPoint::NumericalPoint(const UnsignedLong size,
								 const NumericalScalar value)
	: PersistentCollection<NumericalScalar>(size, value) //,
	  // p_description_()
      {
        // Nothing to do
      }



      /* Constructor from a collection */
      NumericalPoint::NumericalPoint(const Type::Collection<NumericalScalar> & coll)
	: PersistentCollection<NumericalScalar>(coll) //,
	  // p_description_()
      {
        // Nothing to do
      }



      /* Virtual constructor */
      NumericalPoint * NumericalPoint::clone() const
      {
	return new NumericalPoint(*this);
      }


      /* Destructor */
      NumericalPoint::~NumericalPoint() throw()
      {
	// Nothing to do
      }


      /* Coordinate accessor */
      NumericalScalar & NumericalPoint::operator[](const UnsignedLong index)
      {
	return PersistentCollection<NumericalScalar>::operator[](index);
      }

      /* Coordinate accessor */
      const NumericalScalar & NumericalPoint::operator[](const UnsignedLong index) const
      {
	return PersistentCollection<NumericalScalar>::operator[](index);
      }


      /* String converter */
      String NumericalPoint::__repr__() const
      {
	return OSS() << "class=" << NumericalPoint::GetClassName()
		     << " name=" << getName()
		     << " dimension=" << getDimension()
		     << " values=" << PersistentCollection<NumericalScalar>::__repr__();
      }

      String NumericalPoint::__str__() const
      {
	return PersistentCollection<NumericalScalar>::__str__();
      }


//       /* Description Accessor */
//       void NumericalPoint::setDescription(const Description & description)
//       {
// 	p_description_ = description.get();
//       }


//       /* Description Accessor */
//       Description NumericalPoint::getDescription() const
//       {
// 	return p_description_.isNull() ? Description(getDimension()) : *p_description_;
//       }


      

      /* Erase the elements between first and last */
      NumericalPoint::iterator NumericalPoint::erase(iterator first, iterator last)
      {
	return PersistentCollection<NumericalScalar>::erase(first, last);
      }
      
      /* Erase the element pointed by position */
      NumericalPoint::iterator NumericalPoint::erase(iterator position)
      {
	return PersistentCollection<NumericalScalar>::erase(position);
      }
      
      /* Erase the element pointed by position */
      NumericalPoint::iterator NumericalPoint::erase(UnsignedLong position)
      {
	return PersistentCollection<NumericalScalar>::erase(begin() + position);
      }
      


      /* Addition operator */
      NumericalPoint NumericalPoint::operator +(const NumericalPoint & rhs) const
	/* throw (InvalidArgumentException) */
      {
	if (getDimension() != rhs.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPoints of different dimensions cannot be added (LHS dimension = "
	    << getDimension()
	    << "; RHS dimension = "
	    << rhs.getDimension();
	

	// We create a NumericalPoint of the same dimension as both points for holding the result
	NumericalPoint result(getDimension());
	for(UnsignedLong i=0; i<getDimension(); ++i) result[i] = (*this)[i] + rhs[i];
	return result;
      }



      /* In-place addition operator */
      NumericalPoint & NumericalPoint::operator +=(const NumericalPoint & other)
 	/* throw (InvalidArgumentException) */
      {
	if (getDimension() != other.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPoints of different dimensions cannot be added (LHS dimension = "
	    << getDimension()
	    << "; RHS dimension = "
	    << other.getDimension();
	

	for (UnsignedLong i = 0; i < getDimension(); ++i) operator[](i) += other[i];
	return *this;
      }
      


      /* Substraction operator */
      NumericalPoint NumericalPoint::operator -(const NumericalPoint & rhs) const
 	/* throw (InvalidArgumentException) */
      {
	if (getDimension() != rhs.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPoints of different dimensions cannot be substracted (LHS dimension = "
	    << getDimension()
	    << "; RHS dimension = "
	    << rhs.getDimension();


	// We create a NumericalPoint of the same dimension as both points for holding the result
	NumericalPoint result(getDimension());
	for(UnsignedLong i=0; i<getDimension(); ++i) result[i] = (*this)[i] - rhs[i];
	return result;
      }



      /* In-place substraction operator */
      NumericalPoint & NumericalPoint::operator -=(const NumericalPoint & other)
 	/* throw (InvalidArgumentException) */
      {
	if (getDimension() != other.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPoints of different dimensions cannot be substracted (LHS dimension = "
	    << getDimension()
	    << "; RHS dimension = " <<
	    other.getDimension();
	

	for(UnsignedLong i=0; i<getDimension(); ++i) (*this)[i] -= other[i];
	return *this;
      }
      


      /* Product operator */
      NumericalPoint NumericalPoint::operator *(const NumericalScalar scalar) const
      {
	// We create a NumericalPoint of the same dimension as both points for holding the result
	NumericalPoint result(getDimension());
	for(UnsignedLong i=0; i<getDimension(); ++i) result[i] = scalar * (*this)[i];
	return result;
      }

      /*  In-place product operator */
      NumericalPoint & NumericalPoint::operator *=(const NumericalScalar scalar)
      {
	for(UnsignedLong i=0; i<getDimension(); ++i) (*this)[i] *= scalar;
	return *this;
      }






      /* Product operator */
      NumericalPoint operator *(const NumericalScalar scalar,
					      const NumericalPoint & point)
      {
	// We create a NumericalPoint of the same dimension as point for holding the result
	NumericalPoint result(point.getDimension());
	for(UnsignedLong i=0; i<point.getDimension(); ++i) result[i] = scalar * point[i];
	return result;
      }



      /* Dot product operator */
      NumericalScalar NumericalPoint::dot(const NumericalPoint & lhs,
							const NumericalPoint & rhs)
	/* throw (InvalidArgumentException) */
      {
	if (lhs.getDimension() != rhs.getDimension())
	  throw InvalidArgumentException(HERE)
	    << "NumericalPoints of different dimensions cannot be added (LHS dimension = "
	    << lhs.getDimension()
	    << "; RHS dimension = " 
	    << rhs.getDimension();


	NumericalScalar dotProduct(0.);
	for(UnsignedLong i=0; i<lhs.getDimension(); ++i) dotProduct += lhs[i] * rhs[i];
	return dotProduct;
      }


      /* Comparison operator */
      Bool operator ==(const NumericalPoint & lhs,
		       const NumericalPoint & rhs)
      {
	return static_cast<const PersistentCollection<NumericalScalar> >(lhs) == static_cast<const PersistentCollection<NumericalScalar> >(rhs);
      }



      /* Ordering operator */
      Bool operator <(const NumericalPoint & lhs,
		      const NumericalPoint & rhs)
      {
	return static_cast<const PersistentCollection<NumericalScalar> >(lhs) < static_cast<const PersistentCollection<NumericalScalar> >(rhs);
      }



      /*  Norm */
      NumericalScalar NumericalPoint::norm() const
      {
	return sqrt( norm2() );
      }
	

      /*  Norm^2 */
      NumericalScalar NumericalPoint::norm2() const
      {
	return NumericalPoint::dot(*this, *this);
      }
	

      /* Method save() stores the object through the StorageManager */
      void NumericalPoint::save(StorageManager::Advocate & adv) const
      {
	PersistentCollection<NumericalScalar>::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void NumericalPoint::load(StorageManager::Advocate & adv)
      {
	PersistentCollection<NumericalScalar>::load(adv);
      }


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
