//                                               -*- C++ -*-
/**
 *  @file  RandomVector.cxx
 *  @brief The class that implements all random vectors
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: RandomVector.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "RandomVector.hxx"
#include "ConstantRandomVector.hxx"
#include "UsualRandomVector.hxx"
#include "CompositeRandomVector.hxx"
#include "EventRandomVectorImplementation.hxx"
#include "FunctionalChaosRandomVector.hxx"
#include "ComparisonOperatorImplementation.hxx"


namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      CLASSNAMEINIT(RandomVector);

      /* Default constructor */
      RandomVector::RandomVector(const RandomVectorImplementation & implementation,
				 const String & name)
	: Base::Common::TypedInterfaceObject<RandomVectorImplementation>(implementation.clone())
      {
	getImplementation()->setName(name);
      }

      /* Constructor from implementation */
      RandomVector::RandomVector(const Implementation & p_implementation,
				 const String & name)
	: Base::Common::TypedInterfaceObject<RandomVectorImplementation>(p_implementation)
      {
	getImplementation()->setName(name);
      }

      /* Constructor from implementation pointer */
      RandomVector::RandomVector(RandomVectorImplementation * p_implementation,
				 const String & name)
	: Base::Common::TypedInterfaceObject<RandomVectorImplementation>(p_implementation)
      {
	getImplementation()->setName(name);
      }

      /* Constructor for constant vector */
      RandomVector::RandomVector(const NumericalPoint & point,
				 const String & name)
	: Base::Common::TypedInterfaceObject<RandomVectorImplementation>(new ConstantRandomVector(point, name))
      {
	// Nothing to do
      }

      /* Constructor for distribution-based vector */
      RandomVector::RandomVector(const Distribution & distribution,
				 const String & name)
	: Base::Common::TypedInterfaceObject<RandomVectorImplementation>(new UsualRandomVector(distribution, name))
      {
	// Nothing to do
      }

      /* Constructor for composite vector */
      RandomVector::RandomVector(const NumericalMathFunction & function, 
				 const RandomVector & antecedent,
				 const String & name)
 	: Base::Common::TypedInterfaceObject<RandomVectorImplementation>(new CompositeRandomVector(function,
												   antecedent.getImplementation(),
												   name))
      {
	// Nothing to do
      }
     
      /* Constructor for functional chaos vector */
      RandomVector::RandomVector(const FunctionalChaosResult & functionalChaosResult)
 	: Base::Common::TypedInterfaceObject<RandomVectorImplementation>(new FunctionalChaosRandomVector(functionalChaosResult))
      {
	// Nothing to do
      }
     
      /* Constructor from event RandomVector */
      RandomVector::RandomVector(const RandomVector & antecedent,
				 const ComparisonOperator & op,
				 const NumericalScalar threshold,
				 const String & name)
	throw (InvalidArgumentException)
	: Base::Common::TypedInterfaceObject<RandomVectorImplementation>(new EventRandomVectorImplementation(*antecedent.getImplementation(), op, threshold, name))
      {
	// Nothing to do
      }
     
      /* String converter */
      String RandomVector::__repr__() const {
	OSS oss;
	oss << "class=" << RandomVector::GetClassName()
	    << " implementation=" << getImplementation()->__repr__();
	return oss;
      }
  


      /* Description Accessor */
      void RandomVector::setDescription(const Description & description)
      {
	getImplementation()->setDescription(description);
      }


      /* Description Accessor */
      RandomVector::Description RandomVector::getDescription() const
      {
	return getImplementation()->getDescription();
      }


      /* Is the underlying random vector composite ? */
      Bool RandomVector::isComposite() const
      {
	return getImplementation()->isComposite();
      }


      /* Here is the interface that all derived class may implement */

      /* Dimension accessor */
      UnsignedLong RandomVector::getDimension() const
      {
	return getImplementation()->getDimension();
      }

      /* Realization accessor */
      RandomVector::NumericalPoint RandomVector::getRealization() const
      {
	return getImplementation()->getRealization();
      }

      /* Numerical sample accessor */
      RandomVector::NumericalSample RandomVector::getNumericalSample(UnsignedLong size) const
      {
	return getImplementation()->getNumericalSample(size);
      }

      /* Mean accessor */
      RandomVector::NumericalPoint RandomVector::getMean() const
      {
	return getImplementation()->getMean();
      }

      /* Covariance accessor */
      RandomVector::CovarianceMatrix RandomVector::getCovariance() const
      {
	return getImplementation()->getCovariance();
      }

      /* Get the random vector corresponding to the i-th marginal component */
      RandomVector RandomVector::getMarginal(const UnsignedLong i) const throw(InvalidArgumentException)
      {
	return *(getImplementation()->getMarginal(i));
      }

      /* Get the random vector corresponding to the i-th marginal component */
      RandomVector RandomVector::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
	return *(getImplementation()->getMarginal(indices));
      }

      /* This method allows to access the antecedent RandomVector in case of a composite RandomVector */
      RandomVector::Antecedent RandomVector::getAntecedent() const
      {
	return getImplementation()->getAntecedent();
      }

      /* This method allows to access the NumericalMathFunction in case of a composite RandomVector */
      RandomVector::NumericalMathFunction RandomVector::getFunction() const
      {
	return getImplementation()->getFunction();
      }

      /* This method allows to access the Distribution in case of a usual RandomVector */
      Distribution RandomVector::getDistribution() const
      {
	return getImplementation()->getDistribution();
      }

      /* Operator accessor */
      RandomVector::ComparisonOperator RandomVector::getOperator() const
      {
	return getImplementation()->getOperator();
      }
      
      /* Threshold accessor */
      NumericalScalar RandomVector::getThreshold() const
      {
	return getImplementation()->getThreshold();
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
