//                                               -*- C++ -*-
/**
 *  @file  CopulaImplementation.cxx
 *  @brief Abstract top-level class for Continuous distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: CopulaImplementation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */

#include <cmath>
#include "CopulaImplementation.hxx"
#include "IndependentCopula.hxx"
#include "Distribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      typedef Uncertainty::Distribution::IndependentCopula IndependentCopula;

      CLASSNAMEINIT(CopulaImplementation);

      /* Default constructor */
      CopulaImplementation::CopulaImplementation(const String & name)
	: ContinuousDistribution(name)
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
      }

      /* Virtual constructor */
      CopulaImplementation * CopulaImplementation::clone() const
      {
	return new CopulaImplementation(*this);
      }

      /* Comparison operator */
      Bool CopulaImplementation::operator ==(const CopulaImplementation & other) const {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String CopulaImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << CopulaImplementation::GetClassName();
	return oss;
      }
  
      /* Generic implementation of the quantile computation for copulas */
      CopulaImplementation::NumericalPoint CopulaImplementation::computeQuantile(const NumericalScalar prob,
										 const Bool tail) const
      {
	if (prob < 0.0 || prob > 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
	// Special case for boarding values
	if (prob == 0.0) return getRange().getLowerBound();
	if (prob == 1.0) return getRange().getUpperBound();

	UnsignedLong dimension(getDimension());
	// Special case for copulas of dimension 1, i.e. uniform distribution over [0,1]
	if (dimension == 1) return NumericalPoint(1, prob);
	// For a copula, the n-D quantile is defined as X = (tau, ..., tau),
	// with tau such as C(X) = prob.
	// We solve this last equation with respect to tau using the secant method
	// Lower end of the bracketing interval
	NumericalScalar leftTau(prob);
	NumericalScalar leftCDF(0.0);
	// Upper end of the bracketing interval
	NumericalScalar rightTau(1.0 - (1.0 - prob) / dimension);
	NumericalScalar rightCDF(1.0);
	// Main loop of the bisection method
	while ((rightCDF - leftCDF > 2.0 * cdfEpsilon_) && (rightTau - leftTau > 2.0 * cdfEpsilon_))
	  {
	    NumericalScalar middleTau(0.5 * (rightTau + leftTau));
	    NumericalScalar middleCDF(computeCDF(NumericalPoint(dimension, middleTau)));
	    if (middleCDF > prob)
	      {
		rightTau = middleTau;
		rightCDF = middleCDF;
	      }
	    else
	      {
		leftTau = middleTau;
		leftCDF = middleCDF;
	      }
	  }
	return NumericalPoint(dimension, 0.5 * (leftTau + rightTau));
      }


      /* Get the mean of the distribution */
      CopulaImplementation::NumericalPoint CopulaImplementation::getMean() const throw(NotDefinedException)
      {
	return NumericalPoint(getDimension(), 0.5);
      }

      /* Get the standard deviation of the distribution */
      CopulaImplementation::NumericalPoint CopulaImplementation::getStandardDeviation() const throw(NotDefinedException)
      {
	// 0.2886751345948128822545744 = 1 / sqrt(12)
	return NumericalPoint(getDimension(), 0.2886751345948128822545744);
      }
      
      /* Get the skewness of the distribution */
      CopulaImplementation::NumericalPoint CopulaImplementation::getSkewness() const throw(NotDefinedException)
      {
	return NumericalPoint(getDimension(), 0.0);
      }

      /* Get the kurtosis of the distribution */
      CopulaImplementation::NumericalPoint CopulaImplementation::getKurtosis() const throw(NotDefinedException)
      {
	// 1.8 = 9/5
	return NumericalPoint(getDimension(), 1.8);
      }

      /* Compute the covariance of the distribution */
      void CopulaImplementation::computeCovariance() const throw(NotDefinedException)
      {
	UnsignedLong dimension(getDimension());
	// We need this to initialize the covariance matrix in two cases:
	// + this is the first call to this routine (which could be checked by testing the dimension of the distribution and the dimension of the matrix
	// + the copula has changed from a non-independent one to the independent copula
	covariance_ = CovarianceMatrix(dimension);
	// First the diagonal terms, which are the marginal covariances
	// To ensure that the mean is up to date
	mean_ = getMean();
	// Compute the weights and nodes off the 1D gauss quadrature over [-1, 1]
	NumericalSample nodesAndWeights(getGaussNodesAndWeights());
	// Convert the nodes and weights for the interval [0, 1]
	for (UnsignedLong i = 0; i < integrationNodesNumber_; ++i)
	  {
	    nodesAndWeights[0][i] = 0.5 * (nodesAndWeights[0][i] + 1.0);
	    nodesAndWeights[1][i] *= 0.5;
	  }
	// Uniform marginals, the diagonal is 1/12
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    // 0.08333333333333333333333333 = 1 / 12
	    covariance_(i, i) = 0.08333333333333333333333333;
	  }
	// Off-diagonal terms if the copula is not the independent copula
	if (!hasIndependentCopula())
	  {
	    // Performs the integration for each covariance in the strictly lower triangle of the covariance matrix
	    // We simply use a product gauss quadrature
	    // We first loop over the coeeficients because the most expensive task is to get the 2D marginal copulas
            Indices indices(2);
	    for(UnsignedLong rowIndex = 0; rowIndex < dimension; ++rowIndex)
	      {
		indices[0] = rowIndex;
		for(UnsignedLong columnIndex = rowIndex + 1; columnIndex < dimension; ++columnIndex)
		  {
		    indices[1] = columnIndex;
		    Distribution marginalDistribution(getMarginal(indices));
		    if (!marginalDistribution.getImplementation()->hasIndependentCopula())
		      {
			NumericalScalar covarianceIJ(0.0);
			// Then we loop over the integration points
			for(UnsignedLong rowNodeIndex = 0; rowNodeIndex < integrationNodesNumber_; ++rowNodeIndex)
			  {
			    NumericalScalar nodeI(nodesAndWeights[0][rowNodeIndex]);
			    NumericalScalar weightI(nodesAndWeights[1][rowNodeIndex]);
			    for(UnsignedLong columnNodeIndex = 0; columnNodeIndex < integrationNodesNumber_; ++columnNodeIndex)
			      {
				NumericalScalar nodeJ(nodesAndWeights[0][columnNodeIndex]);
				NumericalScalar weightJ(nodesAndWeights[1][columnNodeIndex]);
				NumericalPoint in(2);
				in[0] = nodeI;
				in[1] = nodeJ;
				covarianceIJ += weightI * weightJ * (marginalDistribution.computeCDF(in) - nodeI * nodeJ);
			      } // loop over J integration nodes
			  } // loop over I integration nodes
			covariance_(rowIndex, columnIndex) = covarianceIJ;
		      }
		  } // loop over column indices
	      } // loop over row indices
	  } // if !hasIndependentCopula
	isAlreadyComputedCovariance_ = true;
      } // computeCovariance

      /* Get the i-th marginal distribution */
      CopulaImplementation::Implementation CopulaImplementation::getMarginal(const UnsignedLong i) const throw(InvalidArgumentException)
      {
      	if (i >= getDimension()) throw InvalidArgumentException(HERE) << "The index of a marginal distribution must be in the range [0, dim-1]";
	IndependentCopula result(1);
	result.setWeight(getWeight());
      	return result.clone();
      }
      
      /* Get the copula of a distribution */
      CopulaImplementation::Implementation CopulaImplementation::getCopula() const
      {
	return clone();
      }

      /* Get the mathematical and numerical range of the distribution.
	 Its mathematical range is the smallest closed interval outside
	 of which the PDF is zero, and the numerical range is the interval
	 outside of which the PDF is rounded to zero in double precision */
      CopulaImplementation::Interval CopulaImplementation::getRange() const
      {
	return Interval(getDimension());
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
