//                                               -*- C++ -*-
/**
 *  @file  UserDefined.cxx
 *  @brief The UserDefined distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: UserDefined.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cstdlib>
#include <cmath>
#include "UserDefined.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Type {

      using Uncertainty::Distribution::UserDefined;
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<UserDefined::Pair>);
      static Common::Factory<PersistentCollection<UserDefined::Pair> > RegisteredFactory("PersistentCollection<UserDefined::Pair>");

    } /* namespace Type */
  } /* namespace Base */




  namespace Uncertainty {

    namespace Distribution {


      typedef Base::Stat::RandomGenerator              RandomGenerator;

      /* Sub-classes methods */

      CLASSNAMEINIT(UserDefined::Pair);
      static Base::Common::Factory<UserDefined::Pair> RegisteredFactory_alt1("UserDefined::Pair");

      // Comparison operator for Pair objects
      Bool operator == (const UserDefined::Pair & lhs, const UserDefined::Pair & rhs)
      {
        return (lhs.getX() == rhs.getX()) && (lhs.getP() == rhs.getP());
      }


      CLASSNAMEINIT(UserDefined);
      static Base::Common::Factory<UserDefined> RegisteredFactory_alt2("UserDefined");


      /* Default constructor */
      UserDefined::UserDefined()
        : DiscreteDistribution("UserDefined"),
          collection_(PairCollection(1, Pair(NumericalPoint(1, 0.0), 1.0))),
          cumulativeProbabilities_(NumericalScalarCollection(1, 1.0))
      {
        setRange(Interval(0.0, 0.0));
      }

      /* Constructor from PairCollection */
      UserDefined::UserDefined(const PairCollection & collection)
        throw (InvalidArgumentException)
        : DiscreteDistribution("UserDefined"),
          collection_(),
          cumulativeProbabilities_()
      {
        // We set the dimension of the UserDefined distribution
        setPairCollection( collection );
      }

      /* Comparison operator */
      Bool UserDefined::operator ==(const UserDefined & other) const
      {
        Bool sameObject = false;

        if (this != &other) { // Other is NOT me, so I have to realize the comparison
          if ( collection_ == other.collection_ )
            sameObject = true;

        } else sameObject = true;

        return sameObject;
      }

      /* String converter */
      String UserDefined::__repr__() const
      {
        OSS oss;
        oss << "class=" << UserDefined::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " collection=" << collection_;
        return oss;
      }

      /* Virtual constructor */
      UserDefined * UserDefined::clone() const
      {
        return new UserDefined(*this);
      }

      /* Get one realization of the distribution */
      UserDefined::NumericalPoint UserDefined::getRealization() const
      {
        const NumericalScalar uniformRealization(RandomGenerator::Generate());
        UnsignedLong j(0);
        while(uniformRealization > cumulativeProbabilities_[j]) ++j;
        return collection_[j].getX();
      }

      /* Get the PDF of the distribution */
      NumericalScalar UserDefined::computePDF(const NumericalPoint & point) const
      {
        const UnsignedLong size(collection_.getSize());
        NumericalScalar pdf(0.0);
        // We go through all the points in order to cumulate
        // the contributions of a multiply defined point
        for (UnsignedLong i = 0; i < size; ++i)
          {
            if ((point - collection_[i].getX()).norm() < DiscreteDistribution::SupportEpsilon) pdf += collection_[i].getP();
          }
        return pdf;
      }

      /* Get the CDF of the distribution */
      NumericalScalar UserDefined::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        NumericalScalar cdf(0.0);
        const UnsignedLong size(collection_.getSize());
        const UnsignedLong dimension(getDimension());
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const NumericalPoint x(collection_[i].getX());
            UnsignedLong j(0);
            if (tail)
              {
                while ((j < dimension) && (x[j] <= point[j] + DiscreteDistribution::SupportEpsilon)) ++j;
              }
            else
              {
                while ((j < dimension) && (x[j] > point[j] + DiscreteDistribution::SupportEpsilon)) ++j;
              }
            if (j == dimension) cdf += collection_[i].getP();
          }
        return cdf;
      }

      /* Get the PDF gradient of the distribution */
      UserDefined::NumericalPoint UserDefined::computePDFGradient(const NumericalPoint & point) const
      {
        const UnsignedLong size(collection_.getSize());
        NumericalPoint pdfGradient(size, 0.0);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            if ((point - collection_[i].getX()).norm() < DiscreteDistribution::SupportEpsilon)
              {
                pdfGradient[i] = 1.0;
                return pdfGradient;
              }
          }
        return pdfGradient;
      }


      /* Get the CDF gradient of the distribution */
      UserDefined::NumericalPoint UserDefined::computeCDFGradient(const NumericalPoint & point) const
      {
        const UnsignedLong size(collection_.getSize());
        const UnsignedLong dimension(getDimension());
        NumericalPoint cdfGradient(size, 0.0);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const NumericalPoint x(collection_[i].getX());
            UnsignedLong j(0);
            while ((j < dimension) && (point[j] <= x[j])) ++j;
            if (j == dimension) cdfGradient[i] = 1.0;
          }
        return cdfGradient;
      }



      /* Get the mean of the distribution */
      UserDefined::NumericalPoint UserDefined::getMean() const throw(NotDefinedException)
      {
        const UnsignedLong size(collection_.getSize());
        NumericalPoint mean(getDimension());
        for (UnsignedLong i = 0; i < size; ++i)
          {
            mean += collection_[i].getP() * collection_[i].getX();
          }
        return mean;
      }

      /* Get the covariance of the distribution */
      UserDefined::CovarianceMatrix UserDefined::getCovariance() const throw(NotDefinedException)
      {
        const UnsignedLong size(collection_.getSize());
        const UnsignedLong dimension(getDimension());
        CovarianceMatrix covariance(dimension);
        const NumericalPoint mean(getMean());
        for (UnsignedLong k = 0; k < size; ++k)
          {
            const NumericalPoint xK(collection_[k].getX() - mean);
            const NumericalScalar pK(collection_[k].getP());
            for (UnsignedLong i = 0; i < dimension; ++i)
              {
                for (UnsignedLong j = 0; j <= i; ++j)
                  {
                    covariance(i, j) += pK * xK[i] * xK[j];
                  }
              }
          }
        return covariance;
      }

      /* Parameters value and description accessor */
      UserDefined::NumericalPointWithDescriptionCollection UserDefined::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        const UnsignedLong size(collection_.getSize());
        NumericalPointWithDescription point(size);
        Description description(point.getDimension());
        for (UnsignedLong i = 0; i < size; ++i)
          {
            point[i] = collection_[i].getP();
            OSS oss;
            oss << "p_" << i;
            description[i] = oss;
          }
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      /* Interface specific to UserDefined */

      /* Pair collection accessor */
      void UserDefined::setPairCollection(const PairCollection & collection)
        throw (InvalidArgumentException)
      {
        const UnsignedLong size(collection.getSize());
        if (size == 0) throw InvalidArgumentException(HERE) << "Error: the collection is empty";
        const UnsignedLong dimension(collection[0].getX().getDimension());
        if (dimension == 0) throw InvalidArgumentException(HERE) << "Error: the points in the collection must have a dimension > 0";
        // First, sort the collection such that the sample made with the first component is in ascending order
        NumericalSample firstComponent(size, 1);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            if (collection[i].getX().getDimension() != dimension) throw InvalidArgumentException(HERE) << "UserDefined distribution must have all its point with the same dimension, which is not the case here collection=" << collection;
            firstComponent[i][0] = collection[i].getX()[0];
          }
        setDimension(dimension);
        const NumericalSample index(firstComponent.rank(0));
        PairCollection sortedCollection(collection);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            sortedCollection[(UnsignedLong)(round(index[i][0]))] = collection[i];
          }
        // Check if all the given probabilities are >= 0 and if their sum is 1
        // Check if all the points have the same dimension
        NumericalScalar sum(0.0);
        cumulativeProbabilities_ = NumericalScalarCollection(size);
        for (UnsignedLong i = 0; i < size; ++i)
          {
            const NumericalScalar p(sortedCollection[i].getP());
            if (p < 0.0) throw InvalidArgumentException(HERE) << "UserDefined distribution must have positive probabilities, which is not the case here collection=" << collection;
            sum += p;
            cumulativeProbabilities_[i] = sum;
          }
        if (fabs(sum - 1.0) > DiscreteDistribution::SupportEpsilon) throw InvalidArgumentException(HERE) << "UserDefined distribution must have probabilities that sum to 1.0, which is not the case here collection=" << collection;
        // We augment slightly the last cumulative probability, which should be equal to 1.0 but we enforce a value > 1.0. It stabilize the sampling procedures without affecting their correctness (i.e. the algoritms are exact, not approximative)
        cumulativeProbabilities_[size - 1] = 1.0 + DiscreteDistribution::SupportEpsilon;
        collection_ = sortedCollection;
      }

      UserDefined::PairCollection UserDefined::getPairCollection() const
      {
        return collection_;
      }

      /* Quantile computation for dimension=1 */
      NumericalScalar UserDefined::computeScalarQuantile(const NumericalScalar prob,
                                                         const NumericalScalar initialGuess,
                                                         const NumericalScalar initialStep,
                                                         const NumericalScalar precision) const
      {
        UnsignedLong index(0);
        while (cumulativeProbabilities_[index] < prob) ++index;
        return collection_[index].getX()[0];
      }

      /* Method save() stores the object through the StorageManager */
      void UserDefined::save(const StorageManager::Advocate & adv) const
      {
        Model::DiscreteDistribution::save(adv);
        adv.writeValue(collection_, StorageManager::MemberNameAttribute, "collection_");
        adv.writeValue(cumulativeProbabilities_, StorageManager::MemberNameAttribute, "cumulativeProbabilities_");
      }

      /* Method load() reloads the object from the StorageManager */
      void UserDefined::load(const StorageManager::Advocate & adv)
      {
        Model::DiscreteDistribution::load(adv);
        adv.readValue(collection_, StorageManager::MemberNameAttribute, "collection_");
        adv.readValue(cumulativeProbabilities_, StorageManager::MemberNameAttribute, "cumulativeProbabilities_");
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

