//                                               -*- C++ -*-
/**
 *  @file  Poisson.hxx
 *  @brief The Poisson distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Poisson.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_POISSON_HXX
#define OPENTURNS_POISSON_HXX

#include "OTprivate.hxx"
#include "DiscreteDistribution.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Poisson
       *
       * The Poisson distribution.
       */
      class Poisson
        : public Model::DiscreteDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::DiscreteDistribution                          DiscreteDistribution;  // required by SWIG
        typedef DiscreteDistribution::NumericalPoint                 NumericalPoint;
        typedef DiscreteDistribution::NumericalSample                NumericalSample;
        typedef DiscreteDistribution::CovarianceMatrix               CovarianceMatrix;
        typedef DiscreteDistribution::NumericalPointWithDescriptionCollection       NumericalPointWithDescriptionCollection;
        typedef DiscreteDistribution::NotDefinedException            NotDefinedException;
        typedef DiscreteDistribution::StorageManager                 StorageManager;


        /** Default constructor */
        Poisson();

        /** Parameters constructor */
        Poisson(const NumericalScalar lambda)
          throw (InvalidArgumentException);


        /** Comparison operator */
        Bool operator ==(const Poisson & other) const;

        /** String converter */
	String __repr__() const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual Poisson * clone() const;

        /** Get one realization of the distribution */
        NumericalPoint getRealization() const;

        /** Get the PDF of the distribution */
        using DiscreteDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution */
        using DiscreteDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the PDFGradient of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDFGradient of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Get the quantile of the distribution */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const NumericalScalar initialGuess = 0.0,
                                              const NumericalScalar initialStep = 1.0,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** Get the mean of the distribution */
        NumericalPoint getMean() const throw(NotDefinedException);

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const throw(NotDefinedException);

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const throw(NotDefinedException);

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const throw(NotDefinedException);

        /** Get the covariance of the distribution */
        CovarianceMatrix getCovariance() const throw(NotDefinedException);

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;
        using DiscreteDistribution::setParametersCollection;
        void setParametersCollection(const NumericalPointCollection & parametersCollection);

        /* Interface specific to Poisson */

        /** Lambda accessor */
        void setLambda(const NumericalScalar lambda)
          throw (InvalidArgumentException);
        NumericalScalar getLambda() const;

        /** Method save() stores the object through the StorageManager */
        void save(const StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(const StorageManager::Advocate & adv);

      protected:


      private:

        /** The Lambda of the Poisson distribution */
        NumericalScalar lambda_;

      }; /* class Poisson */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_POISSON_HXX */
