//                                               -*- C++ -*-
/**
 *  @file  Mixture.hxx
 *  @brief The class that implements mixtures
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Mixture.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_MIXTURE_HXX
#define OPENTURNS_MIXTURE_HXX

#include "Distribution.hxx"
#include "DistributionImplementation.hxx"
#include "Exception.hxx"
#include "Collection.hxx"
#include "UserDefined.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Mixture
       *
       * The class describes the probabilistic concept of Mixture.
       */
      class Mixture 
	: public Model::DistributionImplementation
      {
	CLASSNAME;
      public:

	/** A type for distribution collection */
	typedef Base::Type::Collection<Model::Distribution>          DistributionCollection;
	typedef Model::DistributionImplementation                    DistributionImplementation;  // required by SWIG
	typedef DistributionImplementation::InvalidArgumentException InvalidArgumentException;
	typedef DistributionImplementation::NumericalPoint           NumericalPoint;
	typedef DistributionImplementation::NumericalSample          NumericalSample;
	typedef UserDefined::Pair                                    Pair;
	typedef UserDefined::PairCollection                          PairCollection;
	typedef DistributionImplementation::Implementation           Implementation;
	typedef DistributionImplementation::Indices                  Indices;
	typedef DistributionImplementation::CovarianceMatrix         CovarianceMatrix;
	typedef DistributionImplementation::NotDefinedException      NotDefinedException;
	typedef DistributionImplementation::NumericalPointWithDescriptionCollection NumericalPointWithDescriptionCollection;

	/** Default constructor */
	explicit Mixture(const DistributionCollection & coll)
	  throw (InvalidArgumentException);


	/** Comparison operator */
	Bool operator ==(const Mixture & other) const;

	/** String converter */
	String __repr__() const;


	/** Distribution collection accessor */
	void setDistributionCollection(const DistributionCollection & coll)
	  throw (InvalidArgumentException);
	const DistributionCollection & getDistributionCollection() const;



	/* Here is the interface that all derived class must implement */

	/** Virtual constructor */
	virtual Mixture * clone() const;

	/** Get one realization of the Mixture */
	NumericalPoint getRealization() const;

	/** Get the DDF of the Mixture */
	using DistributionImplementation::computeDDF;
	NumericalPoint computeDDF(const NumericalPoint & point) const;

	/** Get the PDF of the Mixture */
	using DistributionImplementation::computePDF;
	NumericalScalar computePDF(const NumericalPoint & point) const;

	/** Get the CDF of the Mixture */
	using DistributionImplementation::computeCDF;
	NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

	/** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
	NumericalComplex computeCharacteristicFunction(const NumericalScalar x,
						       const Bool logScale = false) const;

	/** Get the PDF gradient of the distribution */
	NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Get the CDF gradient of the distribution */
	NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

	/** Get the i-th marginal distribution */
	Implementation getMarginal(const UnsignedLong i) const throw(InvalidArgumentException);

	/** Get the distribution of the marginal distribution corresponding to indices dimensions */
	Implementation getMarginal(const Indices & indices) const throw(InvalidArgumentException);

	/** Get the mean of a mixture */
	NumericalPoint getMean() const throw(NotDefinedException);

	/** Get the covariance of a mixture */
	CovarianceMatrix getCovariance() const throw(NotDefinedException);

	/** Parameters value and description accessor */
	NumericalPointWithDescriptionCollection getParametersCollection() const;

	/** Check if the distribution is elliptical */
        Bool isElliptical() const;

	/** Check if the distribution is constinuous */
        Bool isContinuous() const;

	/** Tell if the distribution has elliptical copula */
	Bool hasEllipticalCopula() const;

	/** Tell if the distribution has independent copula */
	Bool hasIndependentCopula() const;

      protected:

    
      private:

	/** Compute the numerical range of the distribution given the parameters values */
	void computeRange();

	/** Weights distribution accessor */
	void setWeightsDistribution(const UserDefined & weighstDistribution);
	UserDefined getWeightsDistribution() const;

	/** The collection of distribution of the mixture */
	DistributionCollection distributionCollection_;

	/** The discrete distribution of the weights */
	UserDefined weightsDistribution_;
    
      }; /* class Mixture */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_MIXTURE_HXX */
