//                                               -*- C++ -*-
/**
 *  @file  Mixture.cxx
 *  @brief Abstract top-level class for all Mixtures
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Mixture.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cmath>
#include "Mixture.hxx"
#include "Log.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Common::Log Log;

      CLASSNAMEINIT(Mixture);

      /* Default constructor */
      Mixture::Mixture(const DistributionCollection & coll)
	throw (InvalidArgumentException)
	: DistributionImplementation("Mixture"),
	  distributionCollection_(),
	  weightsDistribution_(UserDefined(Mixture::PairCollection(coll.getSize(), Pair(NumericalPoint(1), 1.0 / coll.getSize()))))
      {
	// We could NOT set distributionCollection_ in the member area of the constructor
	// because we must check before if the collection is valid (ie, if all the
	// distributions of the collection have the same dimension). We do this by calling
	// the setDistributionCollection() method that do it for us.
	setDistributionCollection( coll );
      }

      /* Comparison operator */
      Bool Mixture::operator ==(const Mixture & other) const {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String Mixture::__repr__() const {
	OSS oss;
	oss << "class=" << Mixture::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension();
	return oss;
      }
  


      /* Compute the numerical range of the distribution given the parameters values */
      void Mixture::computeRange()
      {
	const UnsignedLong size(distributionCollection_.getSize());
	if (size == 0) return;
	const UnsignedLong dimension(getDimension());
	NumericalPoint lowerBound(dimension);
	NumericalPoint upperBound(dimension);
	Interval::BoolCollection finiteLowerBound(dimension, true);
	Interval::BoolCollection finiteUpperBound(dimension, true);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    const Interval atomRange(distributionCollection_[i].getRange());
	    const NumericalPoint atomLowerBound(atomRange.getLowerBound());
	    const NumericalPoint atomUpperBound(atomRange.getUpperBound());
	    const Interval::BoolCollection atomFiniteLowerBound(atomRange.getFiniteLowerBound());
	    const Interval::BoolCollection atomFiniteUpperBound(atomRange.getFiniteUpperBound());
	    for (UnsignedLong j = 0; j < dimension; ++j)
	      {
		lowerBound[j] = std::min(lowerBound[j], atomLowerBound[j]);
		upperBound[j] = std::max(upperBound[j], atomUpperBound[j]);
		finiteLowerBound[j] = finiteLowerBound[j] && atomFiniteLowerBound[j];
		finiteUpperBound[j] = finiteUpperBound[j] && atomFiniteUpperBound[j];
	      }
	  }
	setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Weights distribution accessor */
      void Mixture::setWeightsDistribution(const UserDefined & weightsDistribution)
      {
	weightsDistribution_ = weightsDistribution;
      }

      UserDefined Mixture::getWeightsDistribution() const
      {
	return weightsDistribution_;
      }

      
      /* Distribution collection accessor */
      void Mixture::setDistributionCollection(const DistributionCollection & coll)
	throw (InvalidArgumentException)
      {
	const UnsignedLong size(coll.getSize());
	if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Mixture based on an empty distribution collection.";
	UnsignedLong dimension(0);
	NumericalScalar weightSum(0.0);
	Bool firstTime(true);
	distributionCollection_ = DistributionCollection(0);
	for(UnsignedLong i = 0; i < size; ++i, firstTime = false)
	  {
	    if (firstTime) dimension = coll[i].getDimension();
	    if (dimension != coll[i].getDimension())
	      // We throw an exception because the collection has distributions of different sizes
	      throw InvalidArgumentException(HERE) << "Collection of distributions has distributions of different dimensions";
	    NumericalScalar w(coll[i].getWeight());
	    if (w < 0.0) throw InvalidArgumentException(HERE) << "Distribution " << i << " has a negative weight, w=" << w;
	    if (w == 0.0)
	      {
		Log::Info(OSS() << "Warning! The distribution number " << i << " has a null weight: it is removed from the collection.");
	      }
	    else
	      {
		distributionCollection_.add(coll[i]);
		weightSum += w;
	      }
	  } /* end for */

	if (weightSum == 0.)
	  // We throw an exception because the collection of distributions has only distributions with null weight: they cannot be renormalized
	  throw InvalidArgumentException(HERE) << "Collection of distributions has only distribution with null weights";

	// We set the member with the collection passed as argument and we renormalize it in place
	PairCollection pairCollection(size, Pair(NumericalPoint(1, 0.0), 0.0));
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    NumericalScalar normalizedWeight(distributionCollection_[i].getWeight() / weightSum);
	    distributionCollection_[i].setWeight(normalizedWeight);
	    pairCollection[i] = Pair(NumericalPoint(1, NumericalScalar(i)), normalizedWeight);
	  } /* end for */
	setWeightsDistribution(UserDefined(pairCollection));
	setDimension(dimension);
	computeRange();
      }


      /* Distribution collection accessor */
      const Mixture::DistributionCollection & Mixture::getDistributionCollection() const
      {
	return distributionCollection_;
      }

      /* Virtual constructor */
      Mixture * Mixture::clone() const
      {
	return new Mixture(*this);
      }


      /* Get one realization of the Mixture */
      Mixture::NumericalPoint Mixture::getRealization() const
      {
	// Select the atom following the weightsDistribution
	UnsignedLong index((UnsignedLong)(round(weightsDistribution_.getRealization()[0])));
	return distributionCollection_[index].getRealization();
      }
     

      /* Get the DDF of the Mixture */
      Mixture::NumericalPoint Mixture::computeDDF(const NumericalPoint & point) const 
      {
	NumericalPoint ddfValue(getDimension(), 0.0);
	if (!getRange().isNumericallyInside(point)) return ddfValue;
	UnsignedLong size(distributionCollection_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    ddfValue += distributionCollection_[i].getWeight() * distributionCollection_[i].computeDDF(point);
	  } /* end for */
	return ddfValue;
      }

      /* Get the PDF of the Mixture */
      NumericalScalar Mixture::computePDF(const NumericalPoint & point) const 
      {
	NumericalScalar pdfValue(0.0);
	const UnsignedLong size(distributionCollection_.getSize());
	if (!getRange().isNumericallyInside(point)) return pdfValue;
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    pdfValue += distributionCollection_[i].getWeight() * distributionCollection_[i].computePDF(point);
	  } /* end for */
	return pdfValue;
      }

      /* Get the CDF of the Mixture */
      NumericalScalar Mixture::computeCDF(const NumericalPoint & point, const Bool tail) const 
      {
	NumericalScalar cdfValue(0.0);
	UnsignedLong size(distributionCollection_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    cdfValue += distributionCollection_[i].getWeight() * distributionCollection_[i].computeCDF(point, tail);
	  } /* end for */
	return cdfValue;
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Mixture::computeCharacteristicFunction(const NumericalScalar x,
							      const Bool logScale) const
      {
	NumericalComplex cfValue(0.0);
	UnsignedLong size(distributionCollection_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    cfValue += distributionCollection_[i].getWeight() * distributionCollection_[i].computeCharacteristicFunction(x);
	  } /* end for */
	if (logScale) return log(cfValue);
	return cfValue;
      }

      /* Get the PDF gradient of the distribution */
      Mixture::NumericalPoint Mixture::computePDFGradient(const NumericalPoint & point) const
      {
	NumericalPoint pdfGradientValue;
	UnsignedLong size(distributionCollection_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    pdfGradientValue += distributionCollection_[i].getWeight() * distributionCollection_[i].computePDFGradient(point);
	  } /* end for */
	return pdfGradientValue;
      }

      /* Get the CDF gradient of the distribution */
      Mixture::NumericalPoint Mixture::computeCDFGradient(const NumericalPoint & point) const
      {
	NumericalPoint cdfGradientValue(getDimension(), 0.0);
	UnsignedLong size(distributionCollection_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    cdfGradientValue += distributionCollection_[i].getWeight() * distributionCollection_[i].computeCDFGradient(point);
	  } /* end for */
	return cdfGradientValue;
      }

      /* Get the i-th marginal distribution */
      Mixture::Implementation Mixture::getMarginal(const UnsignedLong i) const throw(InvalidArgumentException)
      {
	if (i >= getDimension()) throw InvalidArgumentException(HERE) << "The index of a marginal distribution must be in the range [0, dim-1]";
	// Special case for dimension 1
	if (getDimension() == 1) return clone();
	// General case
	DistributionCollection collection;
	UnsignedLong size(distributionCollection_.getSize());
	for (UnsignedLong index = 0; index < size; ++index)
	  {
	    collection.add(distributionCollection_[index].getMarginal(i));
	  }
	Mixture mixture(collection);
	mixture.setName("Marginal of " + getName());
	return mixture.clone();
      }

      /* Get the distribution of the marginal distribution corresponding to indices dimensions */
      Mixture::Implementation Mixture::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
	UnsignedLong dimension(getDimension());
	if (!indices.check(dimension - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal distribution must be in the range [0, dim-1] and  must be different";
	// Special case for dimension 1
	if (dimension == 1) return clone();
	DistributionCollection collection;
	UnsignedLong size(distributionCollection_.getSize());
	for (UnsignedLong index = 0; index < size; ++index)
	  {
	    collection.add(distributionCollection_[index].getMarginal(indices));
	  }
	Mixture mixture(collection);
	mixture.setName("Marginal of " + getName());
	return mixture.clone();
      }

      /* Get the mean of the Mixture */
      Mixture::NumericalPoint Mixture::getMean() const throw(NotDefinedException)
      {
	NumericalPoint mean(getDimension(), 0.0);
	UnsignedLong size(distributionCollection_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    mean += distributionCollection_[i].getWeight() * distributionCollection_[i].getMean();
	  } /* end for */
	return mean;
      }

      /* Get the covariance of the Mixture */
      Mixture::CovarianceMatrix Mixture::getCovariance() const throw(NotDefinedException)
      {
	UnsignedLong dimension(getDimension());
	CovarianceMatrix covariance(dimension);
	UnsignedLong size(distributionCollection_.getSize());
	// First, compute E(X.X^t)
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    NumericalScalar weightI(distributionCollection_[i].getWeight());
	    CovarianceMatrix covarianceI(distributionCollection_[i].getCovariance());
	    NumericalPoint meanI(distributionCollection_[i].getMean());
	    for(UnsignedLong row = 0; row < dimension; row++)
	      {
		for(UnsignedLong column = 0; column <= row; column++)
		  {
		    covariance(row, column) += weightI * (covarianceI(row, column) + meanI[row] * meanI[column]);
		  }
	      }
	  } /* end for */
	// Then, substract E(X).E(X)^t
	NumericalPoint mean(getMean());
	for(UnsignedLong row = 0; row < dimension; row++)
	  {
	    for(UnsignedLong column = 0; column <= row; column++)
	      {
		covariance(row, column) -= mean[row] * mean[column];
	      }
	  }

	return covariance;
      }

      /** Parameters value and description accessor */
      Mixture::NumericalPointWithDescriptionCollection Mixture::getParametersCollection() const
      {
	const UnsignedLong dimension(getDimension());
	const UnsignedLong size(distributionCollection_.getSize());
	// Special case for dimension=1
	if (dimension == 1)
	  {
	    NumericalPointWithDescriptionCollection parameters(1);
	    Description description;
	    // Form a big NumericalPoint from the parameters of each atom
	    for (UnsignedLong i = 0; i < size; ++i)
	      {
		NumericalPointWithDescription atomParameters(distributionCollection_[i].getParametersCollection()[0]);
		Description atomDescription(atomParameters.getDescription());
		UnsignedLong atomParameterDimension(atomParameters.getDimension());
		// Add the current atom parameters
		for (UnsignedLong j = 0; j < atomParameterDimension; j++)
		  {
		    parameters[0].add(atomParameters[i]);
		    description.add(atomDescription[i]);
		  }
	      }
	    parameters[0].setDescription(description);
	    parameters[0].setName(getName());
	    return parameters;
	  }
	// General case
	NumericalPointWithDescriptionCollection parameters(dimension + 1);
	Description description;
	// First put the marginal parameters
	for (UnsignedLong marginalIndex = 0; marginalIndex < dimension; ++marginalIndex)
	  {
	    // Each marginal distribution must output a collection of parameters of size 1, even if it contains an empty NumericalPoint
	    const NumericalPointWithDescriptionCollection marginalParameters(distributionCollection_[marginalIndex].getParametersCollection());
	    NumericalPointWithDescription point(marginalParameters[0]);
	    point.setName(distributionCollection_[marginalIndex].getName());
	    parameters[marginalIndex] = point;
	  } // marginalIndex

	// Form a big NumericalPoint from the dependence parameters of each atom
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    const NumericalPointWithDescription atomDependenceParameters(distributionCollection_[i].getParametersCollection()[dimension]);
	    Description atomDescription(atomDependenceParameters.getDescription());
	    const UnsignedLong atomParameterDimension(atomDependenceParameters.getDimension());
	    // Add the current atom dependence parameters
	    for (UnsignedLong j = 0; j < atomParameterDimension; j++)
	      {
		parameters[dimension].add(atomDependenceParameters[i]);
		description.add(atomDescription[i]);
	      }
	  }
	parameters[dimension].setDescription(description);
	parameters[dimension].setName("dependence");
	return parameters;
      } // getParametersCollection

      /* Check if the distribution is elliptical */
      Bool Mixture::isElliptical() const
      {
	// If there is only one atom
	if (distributionCollection_.getSize() == 1) return distributionCollection_[0].getImplementation()->isElliptical();
	return false;
      }

      /* Check if the distribution is continuos */
      Bool Mixture::isContinuous() const
      {
	const UnsignedLong size(distributionCollection_.getSize());
	Bool flag(distributionCollection_[0].getImplementation()->isContinuous());
	for (UnsignedLong i = 1; i < size; ++i)
	  {
	    flag = flag && distributionCollection_[i].getImplementation()->isContinuous();
	    if (!flag) return false;
	  }
	return flag;
      }

      /* Tell if the distribution has elliptical copula */
      Bool Mixture::hasEllipticalCopula() const
      {
	// In 1D, all the distributions have an elliptical copula
	if (getDimension() == 1) return true;
	// If there is only one atom, the mixture has the same properties as this atom
	if (getDimension() == 1) return distributionCollection_[0].getImplementation()->hasEllipticalCopula();
	// General case
	return false;
      }

      /* Tell if the distribution has independent copula */
      Bool Mixture::hasIndependentCopula() const
      {
	// In 1D, all the distributions have an independent copula
	if (getDimension() == 1) return true;
	// If there is only one atom, the mixture has the same properties as this atom
	if (getDimension() == 1) return distributionCollection_[0].getImplementation()->hasIndependentCopula();
	// General case
	return false;
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
