//                                               -*- C++ -*-
/**
 *  @file  Logistic.cxx
 *  @brief The Logistic distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Logistic.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cmath>
#include "Logistic.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Logistic);

      static Base::Common::Factory<Logistic> RegisteredFactory("Logistic");

      /* Default constructor */
      Logistic::Logistic()
        : NonEllipticalDistribution("Logistic"),
          alpha_(0.0), beta_(1.0)
      {
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      Logistic::Logistic(const NumericalScalar alpha,
                         const NumericalScalar beta)
        throw (InvalidArgumentException)
        : NonEllipticalDistribution("Logistic"),
          alpha_(alpha), beta_(0.)
      {
        setBeta(beta);
        setDimension( 1 );
        computeRange();
      }

      /* Comparison operator */
      Bool Logistic::operator ==(const Logistic & other) const {
        Bool sameObject = false;

        if (this != &other) { // Other is NOT me, so I have to realize the comparison
          // sameObject = ...
          // TODO: Write Logistic::operator ==(...)
          if ( (beta_ == other.beta_) && (alpha_ == other.alpha_) )
            sameObject = true;
        } else sameObject = true;

        return sameObject;
      }

      /* String converter */
      String Logistic::__repr__() const
      {
        OSS oss;
        oss << "class=" << Logistic::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " alpha=" << alpha_
            << " beta=" << beta_;
        return oss;
      }

      /* Virtual constructor */
      Logistic * Logistic::clone() const
      {
        return new Logistic(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Logistic::computeRange()
      {
        const NumericalScalar delta(beta_ * (SpecFunc::LogMaxNumericalScalar + log(beta_)));
        const NumericalPoint lowerBound(1, alpha_ - delta);
        const NumericalPoint upperBound(1, alpha_ + delta);
        const Interval::BoolCollection finiteLowerBound(1, false);
        const Interval::BoolCollection finiteUpperBound(1, false);
        setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Get one realization of the distribution */
      Logistic::NumericalPoint Logistic::getRealization() const
      {
        NumericalScalar prob(RandomGenerator::Generate());
        return NumericalPoint(1, alpha_ + beta_ * log(prob / (1.0 - prob)));
      }


      /* Get the DDF of the distribution */
      Logistic::NumericalPoint Logistic::computeDDF(const NumericalPoint & point) const
      {
        NumericalScalar expX(exp((point[0] - alpha_) / beta_));
        NumericalScalar betaExpX(beta_ * (1.0 + expX));
        return NumericalPoint(1, beta_ * expX * (1.0 - expX) / (betaExpX * betaExpX * betaExpX));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Logistic::computePDF(const NumericalPoint & point) const
      {
        NumericalScalar expX(exp((point[0] - alpha_) / beta_));
        NumericalScalar betaExpX(beta_ * (1.0 + expX));
        return beta_ * expX / (betaExpX * betaExpX);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Logistic::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        if (tail) return 1.0 / (1.0 + exp((point[0] - alpha_) / beta_));
        return 1.0 - 1.0 / (1.0 + exp((point[0] - alpha_) / beta_));
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Logistic::computeCharacteristicFunction(const NumericalScalar x,
                                                               const Bool logScale) const
      {
        const NumericalScalar piBetaU(M_PI * beta_ * x);
        if (logScale) return NumericalComplex(0.0, x * alpha_) + log(piBetaU) - log(sinh(piBetaU));
        return exp(NumericalComplex(0.0, x * alpha_)) * piBetaU / sinh(piBetaU);
      }

      /* Get the PDFGradient of the distribution */
      Logistic::NumericalPoint Logistic::computePDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar x((point[0] - alpha_) / beta_);
        NumericalScalar expX(exp(x));
        NumericalScalar betaExpX(beta_ * (1.0 + expX));
        NumericalPoint pdfGradient(2);
        pdfGradient[0] = beta_ * expX * (expX - 1.0) / (betaExpX * betaExpX * betaExpX);
        pdfGradient[1] = pdfGradient[0] * x - expX / (betaExpX * betaExpX);
        return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Logistic::NumericalPoint Logistic::computeCDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar x((point[0] - alpha_) / beta_);
        NumericalScalar expX(exp(x));
        NumericalScalar betaExpX(beta_ * (1.0 + expX));
        NumericalPoint cdfGradient(2);
        cdfGradient[0] = -beta_ * expX / (betaExpX * betaExpX);
        cdfGradient[1] = cdfGradient[0] * x;
        return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar Logistic::computeScalarQuantile(const NumericalScalar prob,
                                                      const NumericalScalar initialGuess,
                                                      const NumericalScalar initialStep,
                                                      const NumericalScalar precision) const
      {
        return alpha_ + beta_ * log(prob / (1.0 - prob));
      }

      /* Get the roughness, i.e. the L2-norm of the PDF */
      NumericalScalar Logistic::getRoughness() const
      {
        // 0.1666666666666666666666667 = 1 / 6
        return 0.1666666666666666666666667 / beta_;
      }

      /* Get the mean of the distribution */
      Logistic::NumericalPoint Logistic::getMean() const throw(NotDefinedException)
      {
        return NumericalPoint(1, alpha_);
      }

      /* Get the standard deviation of the distribution */
      Logistic::NumericalPoint Logistic::getStandardDeviation() const throw(NotDefinedException)
      {
        return NumericalPoint(1, getSigma());
      }

      /* Get the skewness of the distribution */
      Logistic::NumericalPoint Logistic::getSkewness() const throw(NotDefinedException)
      {
        return NumericalPoint(1, 0.0);
      }

      /* Get the kurtosis of the distribution */
      Logistic::NumericalPoint Logistic::getKurtosis() const throw(NotDefinedException)
      {
        // 3+6/5 = 4.2
        return NumericalPoint(1, 4.2);
      }

      /* Get the covariance of the distribution */
      Logistic::CovarianceMatrix Logistic::getCovariance() const throw(NotDefinedException)
      {
        CovarianceMatrix covariance(1);
        covariance(0, 0) = SpecFunc::PI2_3 * beta_ * beta_;
        return covariance;
      }

      /* Parameters value and description accessor */
      Logistic::NumericalPointWithDescriptionCollection Logistic::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(2);
        Description description(point.getDimension());
        point[0] = alpha_;
        point[1] = beta_;
        description[0] = "alpha";
        description[1] = "beta";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void Logistic::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = Logistic(parametersCollection[0][0], parametersCollection[0][1]);
      }

      /* Alpha accessor */
      void Logistic::setAlpha(const NumericalScalar alpha)
      {
        alpha_ = alpha;
        computeRange();
      }

      NumericalScalar Logistic::getAlpha() const
      {
        return alpha_;
      }


      /* Beta accessor */
      void Logistic::setBeta(const NumericalScalar beta)
        throw(InvalidArgumentException)
      {
        if (beta <= 0.) throw InvalidArgumentException(HERE) << "Beta MUST be positive, here beta=" << beta;
        beta_ = beta;
        computeRange();
      }

      NumericalScalar Logistic::getBeta() const
      {
        return beta_;
      }



      /* Mu accessor */
      void Logistic::setMu(const NumericalScalar mu)
      {
        alpha_ = mu;
        computeRange();
      }

      NumericalScalar Logistic::getMu() const
      {
        return alpha_;
      }


      /* Sigma accessor */
      void Logistic::setSigma(const NumericalScalar sigma)
        throw(InvalidArgumentException)
      {
        if (sigma <= 0.) throw InvalidArgumentException(HERE) << "Sigma MUST be positive, here sigma=" << sigma;
        beta_ = SpecFunc::SQRT3_PI * sigma;
        computeRange();
      }

      NumericalScalar Logistic::getSigma() const
      {
        return SpecFunc::PI_SQRT3 * beta_;
      }

      /* Method save() stores the object through the StorageManager */
      void Logistic::save(const StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
        adv.writeValue("alpha_", alpha_);
        adv.writeValue("beta_", beta_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Logistic::load(const StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);

        String name;
        NumericalScalar value;
        StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
        for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
          if (objList.readValue(name, value)) {
            if (name == "alpha_") alpha_ = value;
            if (name == "beta_") beta_ = value;
          }
        }
        computeRange();
      }



    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
