//                                               -*- C++ -*-
/**
 *  @file  Laplace.hxx
 *  @brief The Laplace distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-11-23 16:03:50 +0100 (dim, 23 nov 2008) $
 *  Id:      $Id: Laplace.hxx 1026 2008-11-23 15:03:50Z lebrun $
 */
#ifndef OPENTURNS_LAPLACE_HXX
#define OPENTURNS_LAPLACE_HXX

#include "NonEllipticalDistribution.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Laplace
       *
       * The Laplace distribution.
       */
      class Laplace
        : public Model::NonEllipticalDistribution
      {
        CLASSNAME;
      public:

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution;  // required by SWIG
        typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
        typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
        typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
        typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
        typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;
        typedef NonEllipticalDistribution::StorageManager            StorageManager;


        /** Default constructor */
        Laplace();

        /** Parameters constructor */
        Laplace(const NumericalScalar lambda,
                const NumericalScalar mu = 0.0)
          throw(InvalidArgumentException);


        /** Comparison operator */
        Bool operator ==(const Laplace & other) const;

        /** String converter */
	String __repr__() const;



        /* Interface inherited from Distribution */
        /** Virtual constructor */
        virtual Laplace * clone() const;

        /** Get one realization of the distribution */
        NumericalPoint getRealization() const;

        /** Get the DDF of the distribution */
        using NonEllipticalDistribution::computeDDF;
        NumericalPoint computeDDF(const NumericalPoint & point) const;

        /** Get the PDF of the distribution */
        using NonEllipticalDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the distribution */
        using NonEllipticalDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
        NumericalComplex computeCharacteristicFunction(const NumericalScalar x,
                                                       const Bool logScale = false) const;

        /** Get the PDFGradient of the distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDFGradient of the distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Get the quantile of the distribution */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
                                              const NumericalScalar initialGuess = 0.0,
                                              const NumericalScalar initialStep = 1.0,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** Get the mean of the distribution */
        NumericalPoint getMean() const throw(NotDefinedException);

        /** Get the standard deviation of the distribution */
        NumericalPoint getStandardDeviation() const throw(NotDefinedException);

        /** Get the skewness of the distribution */
        NumericalPoint getSkewness() const throw(NotDefinedException);

        /** Get the kurtosis of the distribution */
        NumericalPoint getKurtosis() const throw(NotDefinedException);

        /** Get the covariance of the distribution */
        CovarianceMatrix getCovariance() const throw(NotDefinedException);

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;
        using NonEllipticalDistribution::setParametersCollection;
        void setParametersCollection(const NumericalPointCollection & parametersCollection);

        /* Interface specific to Laplace */

        /** Mu accessor */
        void setMu(const NumericalScalar mu)
          throw(InvalidArgumentException);
        NumericalScalar getMu() const;

        /** Lambda accessor */
        void setLambda(const NumericalScalar lambda);
        NumericalScalar getLambda() const;

        /** Method save() stores the object through the StorageManager */
        void save(const StorageManager::Advocate & adv) const;

        /** Method load() reloads the object from the StorageManager */
        void load(const StorageManager::Advocate & adv);

      protected:


      private:

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** The lamdbda of the Laplace distribution */
        NumericalScalar lambda_;

        /** The mu of the Laplace distribution */
        NumericalScalar mu_;

      }; /* class Laplace */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_LAPLACE_HXX */
