//                                               -*- C++ -*-
/**
 *  @file  Geometric.cxx
 *  @brief The Geometric distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: Geometric.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include <cmath>
#include "Geometric.hxx"
#include "SpecFunc.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Geometric);

      static Base::Common::Factory<Geometric> RegisteredFactory("Geometric");

      /* Default constructor */
      Geometric::Geometric()
        : DiscreteDistribution("Geometric"),
          p_(0.5)
      {
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      Geometric::Geometric(const NumericalScalar p)
        throw (InvalidArgumentException)
        : DiscreteDistribution("Geometric"),
          p_(p)
      {
        if ( (p <= 0.0) || (p > 1.0) ) throw InvalidArgumentException(HERE) << "Error: p must be in ]0, 1], here p=" << p;
        // We set the dimension of the Geometric distribution
        setDimension( 1 );
        computeRange();
      }

      /* Comparison operator */
      Bool Geometric::operator ==(const Geometric & other) const
      {
        Bool sameObject = false;

        if (this != &other) { // Other is NOT me, so I have to realize the comparison
          if ( (p_ == other.p_) )
            sameObject = true;

        } else sameObject = true;

        return sameObject;
      }

      /* String converter */
      String Geometric::__repr__() const
      {
        OSS oss;
        oss << "class=" << Geometric::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " p=" << p_;
        return oss;
      }

      /* Virtual constructor */
      Geometric * Geometric::clone() const
      {
        return new Geometric(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Geometric::computeRange()
      {
        setRange(Interval(0.0, ceil(1.0 - (SpecFunc::LogMaxNumericalScalar + log(p_)) / log(1.0 - p_))));
      }

      /* Get one realization of the distribution */
      Geometric::NumericalPoint Geometric::getRealization() const
      {
        return NumericalPoint(1, ceil(log(RandomGenerator::Generate()) / log(1.0 - p_)));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Geometric::computePDF(const NumericalPoint & point) const
      {
        NumericalScalar k(point[0]);
        if ((k < 1.0 - DiscreteDistribution::SupportEpsilon) || (fabs(k - round(k)) > DiscreteDistribution::SupportEpsilon)) return 0.0;
        return p_ * pow(1.0 - p_, k - 1.0);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Geometric::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        NumericalScalar k(point[0]);
        if (k < 1.0) return (tail ? 1.0 : 0.0);
        if (tail) return pow(1.0 - p_, floor(k));
        return 1.0 - pow(1.0 - p_, floor(k));
      }

      /* Get the PDFGradient of the distribution */
      Geometric::NumericalPoint Geometric::computePDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar k(point[0]);
        if ((k < 1.0 - DiscreteDistribution::SupportEpsilon) || (fabs(k - round(k)) > DiscreteDistribution::SupportEpsilon)) return NumericalPoint(1, 0.0);
        return NumericalPoint(1, (1.0 - k * p_) * pow(1.0 - p_, k - 2.0));
      }

      /* Get the CDFGradient of the distribution */
      Geometric::NumericalPoint Geometric::computeCDFGradient(const NumericalPoint & point) const
      {
        NumericalScalar k(floor(point[0]));
        if ( k < 1.0 ) return NumericalPoint(1, 0.0);
        return NumericalPoint(1, k * pow(1 - p_, k - 1.0));
      }

      /* Get the quantile of the distribution */
      NumericalScalar Geometric::computeScalarQuantile(const NumericalScalar prob,
                                                       const NumericalScalar initialGuess,
                                                       const NumericalScalar initialStep,
                                                       const NumericalScalar precision) const
      {
        return ceil(log(1.0 - prob) / log(1.0 - p_));
      }

      /* Get the mean of the distribution */
      Geometric::NumericalPoint Geometric::getMean() const throw(NotDefinedException)
      {
        return NumericalPoint(1, 1 / p_);
      }

      /* Get the standard deviation of the distribution */
      Geometric::NumericalPoint Geometric::getStandardDeviation() const throw(NotDefinedException)
      {
        return NumericalPoint(1, sqrt(1.0 - p_) / p_);
      }

      /* Get the skewness of the distribution */
      Geometric::NumericalPoint Geometric::getSkewness() const throw(NotDefinedException)
      {
        return NumericalPoint(1, (2.0 - p_) / sqrt(1.0 - p_));
      }

      /* Get the kurtosis of the distribution */
      Geometric::NumericalPoint Geometric::getKurtosis() const throw(NotDefinedException)
      {
        return NumericalPoint(1, 9.0 + p_ * p_ / (1.0 - p_));
      }

      /* Get the covariance of the distribution */
      Geometric::CovarianceMatrix Geometric::getCovariance() const throw(NotDefinedException)
      {
        CovarianceMatrix covariance(1);
        covariance(0, 0) = (1.0 - p_) / (p_ * p_);
        return covariance;
      }

      /* Parameters value and description accessor */
      Geometric::NumericalPointWithDescriptionCollection Geometric::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(1);
        Description description(point.getDimension());
        point[0] = p_;
        description[0] = "p";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void Geometric::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = Geometric(parametersCollection[0][0]);
      }



      /* P accessor */
      void Geometric::setP(const NumericalScalar p)
        throw (InvalidArgumentException)
      {
        if ( (p <= 0.0) || (p > 1.0) ) throw InvalidArgumentException(HERE) << "Error: p must be in ]0, 1], here p=" << p;
        p_ = p;
        computeRange();
      }

      /* P accessor */
      NumericalScalar Geometric::getP() const
      {
        return p_;
      }

      /* Method save() stores the object through the StorageManager */
      void Geometric::save(const StorageManager::Advocate & adv) const
      {
        DiscreteDistribution::save(adv);
        adv.writeValue("p_", p_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Geometric::load(const StorageManager::Advocate & adv)
      {
        DiscreteDistribution::load(adv);

        String name;
        NumericalScalar value;
        StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
        for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
          if (objList.readValue(name, value)) {
            if (name == "p_") p_ = value;
          }
        }
        computeRange();
      }



    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
