//                                               -*- C++ -*-
/**
 *  @file  FrankCopulaFactory.cxx
 *  @brief Factory for FrankCopula distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu, 10 mai 2007) $
 *  Id:      $Id: FrankCopulaFactory.cxx 434 2007-05-10 14:43:31Z dutka $
 */
#include <cstdlib>
#include <cmath>

#include "FrankCopulaFactory.hxx"
#include "FrankCopula.hxx"
#include "SpecFunc.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Common::ResourceMap ResourceMap;

      const NumericalScalar FrankCopulaFactory::ThetaEpsilon = 1.0e-14;//strtod(ResourceMap::GetInstance().get("FrankCopulaFactory-ThetaEpsilon").c_str(), NULL);

      /* Default constructor */
      FrankCopulaFactory::FrankCopulaFactory()
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
      }

      /* Virtual constructor */
      FrankCopulaFactory * FrankCopulaFactory::clone() const
      {
	return new FrankCopulaFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      FrankCopulaFactory::DistributionImplementation * FrankCopulaFactory::buildImplementation(const NumericalSample & sample) const throw(InvalidArgumentException, InternalException)
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a FrankCopula distribution from an empty sample";
	if (sample.getDimension() != 2) throw InvalidArgumentException(HERE) << "Error: cannot build a FrankCopula distribution from a sample of dimension not equal to 2";
	NumericalScalar tau(sample.computeKendallTau().operator()(0, 1));
	if (tau == 1) throw InvalidArgumentException(HERE) << "Error: cannot build a FrankCopula distribution from a sample with Kendall tau equals to 1";
	// Search the value of the Frank copula parameter by numerically inverting the relation:
	// between Kendall's tau and Frank copula's parameter
	NumericalScalar theta(1.0);
	NumericalScalar step(0.5);
	NumericalScalar tauTheta(KendallTauFromParameter(theta));
	// Find a lower bound
	while (tauTheta > tau)
	  {
	    theta -= step;
	    tauTheta = KendallTauFromParameter(theta);
	    step *= 0.5;
	  }
	// Here, tauTheta <= tau, hence theta is a lower bound of the parameter
	NumericalScalar minTheta(theta);
	// Now, look for an upper bound
	// If we started from a value of theta such that tauTheta > tau, theta + 2 * step is an upper bound
	NumericalScalar maxTheta(theta + 2.0 * step);
	// but if step = 0.5, it is because tauTheta was < tau for the initial choice of theta
	if (step < 0.5)
	  {
	    while (tauTheta <= tau)
	      {
		theta += step;
		tauTheta = KendallTauFromParameter(theta);
		step *= 2.0;
	      }
	    // Here, we know that [theta - 0.5 * step, theta] is a bracketing interval
	    minTheta = theta - 0.5 * step;
	    maxTheta = theta;
	  }
	// Bisection
	while (maxTheta - minTheta > ThetaEpsilon)
	  {
	    theta = 0.5 * (maxTheta + minTheta);
	    tauTheta = KendallTauFromParameter(theta);
	    if (tauTheta <= tau)
	      {
		minTheta = theta;
	      }
	    else
	      {
		maxTheta = theta;
	      }
	  }
	return FrankCopula(theta).clone();
      }
      
      FrankCopulaFactory::DistributionImplementation * FrankCopulaFactory::buildImplementation(const NumericalPointCollection & parameters) const throw(InvalidArgumentException, InternalException)
      {
	try {
	  FrankCopula distribution;
	  distribution.setParametersCollection(parameters);
	  return distribution.clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot build a FrankCopula distribution from the given parameters";
	  }
      }

      FrankCopulaFactory::DistributionImplementation * FrankCopulaFactory::buildImplementation() const throw(InvalidArgumentException, InternalException)
      {
	return FrankCopula().clone();
      }

      // Compute Kendall's tau from Frank copula's parameter. It is an increasing function
      NumericalScalar FrankCopulaFactory::KendallTauFromParameter(const NumericalScalar theta) const
      {
	// Third order expansion for small theta, to avoid roundoff for theta=0
	if (fabs(theta) < 8.75677785080733459909609107099e-4) return theta * (1.0 - 0.01 * theta * theta) / 9.0;
	return 1.0 - 4.0 * (1.0 - SpecFunc::Debye(theta, 1)) / theta;
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
