//                                               -*- C++ -*-
/**
 *  @file  SimulationResultImplementation.cxx
 *  @brief Implementation of SimulationResult
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-23 13:46:12 +0200 (ven, 23 mai 2008) $
 *  Id:      $Id: SimulationResultImplementation.cxx 821 2008-05-23 11:46:12Z schueller $
 */
#include <cstdlib>

#include "SimulationResultImplementation.hxx"
#include "PersistentObjectFactory.hxx"
#include "DistFunc.hxx"
#include "Log.hxx"
#include "ResourceMap.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(SimulationResultImplementation);

      typedef Uncertainty::Distribution::DistFunc DistFunc;
      typedef Base::Common::Log                   Log;
      typedef Base::Common::ResourceMap           ResourceMap;

      static Base::Common::Factory<SimulationResultImplementation> RegisteredFactory("SimulationResultImplementation");

      const NumericalScalar SimulationResultImplementation::DefaultConfidenceLevel = 0.95;//strtod(ResourceMap::GetInstance().get("SimulationResultImplementation-DefaultConfidenceLevel").c_str(), NULL);

      /* Default constructor */
      SimulationResultImplementation::SimulationResultImplementation()
	: Base::Common::PersistentObject(),
	  probabilityEstimate_(0.0),
	  varianceEstimate_(0.0),
	  outerSampling_(0),
	  blockSize_(0)
      {
	// Nothing to do
      }

      /* Standard constructor */
      SimulationResultImplementation::SimulationResultImplementation(const NumericalScalar probabilityEstimate,
								     const NumericalScalar varianceEstimate,
								     const UnsignedLong outerSampling,
								     const UnsignedLong blockSize) throw(InvalidArgumentException)
	: PersistentObject(),
	  probabilityEstimate_(probabilityEstimate),
	  varianceEstimate_(varianceEstimate),
	  outerSampling_(outerSampling),
	  blockSize_(blockSize)
      {
	// Check if the probability estimate is within the range [0, 1]
	if ((probabilityEstimate < 0) || (probabilityEstimate > 1)) Log::Info("The probability estimate should be in the range [0, 1]");
	// Check if the variance estimate is >= 0.0
	if (varianceEstimate < 0.0) throw InvalidArgumentException(HERE) << "The variance estimate must be >= 0";
      }

      /* Virtual constructor */
      SimulationResultImplementation * SimulationResultImplementation::clone() const
      {
	return new SimulationResultImplementation(*this);
      }

      /* Probability estimate accessor */
      NumericalScalar SimulationResultImplementation::getProbabilityEstimate() const
      {
	return probabilityEstimate_;
      }

      void SimulationResultImplementation::setProbabilityEstimate(const NumericalScalar probabilityEstimate)
      {
	probabilityEstimate_ = probabilityEstimate;
      }

      /* Variance estimate accessor */
      NumericalScalar SimulationResultImplementation::getVarianceEstimate() const
      {
	return varianceEstimate_;
      }

      void SimulationResultImplementation::setVarianceEstimate(const NumericalScalar varianceEstimate)
      {
	varianceEstimate_ = varianceEstimate;
      }

      /* Coefficient of variation estimate accessor */
      NumericalScalar SimulationResultImplementation::getCoefficientOfVariation() const
      {
	// The usual case: the variance estimate is > 0.0 and the probability estimate is in ]0,1]
	if ((varianceEstimate_ > 0.0) && (probabilityEstimate_ > 0.0) && (probabilityEstimate_ <= 1.0)) return sqrt(varianceEstimate_) / probabilityEstimate_;
	// In all the other cases, return -1.0, waiting for a better strategy
	// when covarianceEstimate_ == 0.0 and probabilityEstimate_ > 0.0
	return -1.0;
      }

      /* Outer sampling accessor */
      UnsignedLong SimulationResultImplementation::getOuterSampling() const
      {
	return outerSampling_;
      }

      void SimulationResultImplementation::setOuterSampling(const UnsignedLong outerSampling) 
      {
	outerSampling_ = outerSampling;
      }

      /* Block size accessor */
      UnsignedLong SimulationResultImplementation::getBlockSize() const
      {
	return blockSize_;
      }

      void SimulationResultImplementation::setBlockSize(const UnsignedLong blockSize)
      {
	blockSize_ = blockSize;
      }

      /* String converter */
      String SimulationResultImplementation::__repr__() const
      {
	OSS oss;
	oss.setPrecision(6);
	oss << std::scientific
	    << "probabilityEstimate=" << probabilityEstimate_
	    << " varianceEstimate=" << varianceEstimate_;

	oss.setPrecision(2)
	  << " coefficient of variation=" << getCoefficientOfVariation()
	  << std::fixed
	  << " confidenceLength(" << DefaultConfidenceLevel << ")=" << std::scientific << getConfidenceLength(DefaultConfidenceLevel) << std::fixed
	  << " outerSampling=" << outerSampling_
	  << " blockSize=" << blockSize_;

	return oss;
      }

      /* Confidence length */
      NumericalScalar SimulationResultImplementation::getConfidenceLength(const NumericalScalar level) const
	throw(InvalidArgumentException)
      {
	// Check if the given level is in ]0, 1[
	if ((level <= 0.0) || (level >= 1.0)) throw InvalidArgumentException(HERE) << "Confidence level must be in ]0, 1[";
	// Create a 1-D standard normal distribution
	NumericalScalar xq(DistFunc::qNormal(1.0 - 0.5 * (1.0 - level)));
	return 2.0 * xq * sqrt(varianceEstimate_);
      }

      /* Method save() stores the object through the StorageManager */
      void SimulationResultImplementation::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.writeValue("probabilityEstimate_", probabilityEstimate_);
	adv.writeValue("varianceEstimate_", varianceEstimate_);
	adv.writeValue("outerSampling_", outerSampling_);
	adv.writeValue("blockSize_", blockSize_);
      }

      /* Method load() reloads the object from the StorageManager */
      void SimulationResultImplementation::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
	String name;
	NumericalScalar scalarValue;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, scalarValue)) {
	    if (name == "probabilityEstimate_") probabilityEstimate_ = scalarValue;
	    if (name == "varianceEstimate_") varianceEstimate_ = scalarValue;
	  }
	}
	UnsignedLong unsignedLongValue;
	objList = adv.getList(StorageManager::UnsignedLongEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, unsignedLongValue)) {
	    if (name == "outerSampling_") outerSampling_ = unsignedLongValue;
	    if (name == "blockSize_") blockSize_ = unsignedLongValue;
	  }
	}
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
